using System;
using System.Collections.Generic;
using System.Text;
using System.IO;

namespace AnalysisScopeGenerator
{
    class Program
    {
        public static int Main(String[] args)
        {
            if ((args.Length < 3) || (args.Length > 6))
            {
                return ShowHelpMenu();
            }
            else
            {
                IDictionary<String, String> namedArguments = LoadArguments(args);
                if ((namedArguments == null) || (!ValidateArguments(namedArguments)))
                {
                    return ShowHelpMenu();
                }
                else
                {
                    String scopeFileName;
                    List<String> appAssemblies = new List<string>();
                    List<String> refAssemblies = new List<string>();
                    String sysDLL;
                    String metaPath;

                    if (namedArguments.ContainsKey("precomp"))
                    {
                        DirectoryInfo dir = PreCompile(namedArguments["precomp"]);
                        if (dir == null)
                        {
                            return ShowHelpMenu();
                        }
                        else
                        {
                            appAssemblies.AddRange(FindAssemblies(dir.FullName));
                        }
                    }

                    scopeFileName = namedArguments["target"];
                    sysDLL = namedArguments["sys"];

                    if (namedArguments.ContainsKey("files"))
                    {
                        String rawFiles = namedArguments["files"];
                        appAssemblies.AddRange(rawFiles.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries));
                    }
                    else if (namedArguments.ContainsKey("paths"))
                    {
                        String rawPaths = namedArguments["paths"];
                        appAssemblies.AddRange(FindAssemblies(rawPaths));
                    }

                    metaPath = namedArguments.ContainsKey("meta") ? namedArguments["meta"] : null;

                    if (namedArguments.ContainsKey("refs"))
                    {
                        String refPaths = namedArguments["refs"];
                        refAssemblies.AddRange(FindAssemblies(refPaths));
                    }

                    Factory.MakeScope(scopeFileName, metaPath, sysDLL, appAssemblies.ToArray(), refAssemblies.ToArray());
                    Console.WriteLine("scope created successfully!");
                    return 0;
                }
            }
        }

        private static IDictionary<String, String> LoadArguments(String[] args)
        {
            IDictionary<String, String> result = new Dictionary<String, String>(args.Length);
            foreach (String arg in args)
            {
                if (!(arg.StartsWith("-") && arg.Contains(":")))
                {
                    Console.WriteLine(new StringBuilder("ERROR: Illegal argument format. Arguments must be in the format \"-<argname>:<argval>\"").ToString());
                    return null;
                }
                int delimiterIndex  = arg.IndexOf(':');
                String argName      = arg.Substring(1, delimiterIndex - 1);
                String argValue     = arg.Substring(delimiterIndex + 1);
                result.Add(argName, argValue);
            }
            return result;
        }

        private static bool ValidateArguments(IDictionary<String, String> args)
        {
            return (args.ContainsKey("target") && args.ContainsKey("sys") &&
                (args.ContainsKey("files") || args.ContainsKey("paths") || args.ContainsKey("precomp")));
        }

        private static DirectoryInfo PreCompile(String projectHome)
        {
            if (!Directory.Exists(projectHome))
            {
                Console.WriteLine(new StringBuilder("ERROR: Precompilation path not found: \"").Append(projectHome).Append("\".").ToString());
                return null;
            }
            
            // Create the destination directory to precompile into.
            string path = System.IO.Path.GetTempFileName();
            File.Delete(path);
            DirectoryInfo dir = Directory.CreateDirectory(path);

            System.Web.Compilation.ClientBuildManagerParameter cbmp = new System.Web.Compilation.ClientBuildManagerParameter();
            cbmp.PrecompilationFlags = (System.Web.Compilation.PrecompilationFlags.ForceDebug | System.Web.Compilation.PrecompilationFlags.FixedNames);
            System.Web.Compilation.ClientBuildManager cbm = new System.Web.Compilation.ClientBuildManager("/", projectHome, dir.FullName, cbmp);
            cbm.PrecompileApplication();

            return dir;
        }

        private static List<String> FindAssemblies(String pathsString)
        {
            String[] paths = pathsString.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
            System.Collections.Generic.List<String> result = new System.Collections.Generic.List<String>(paths.Length);
            foreach (String p in paths)
            {
                String[] files = System.IO.Directory.GetFiles(p, "*.*", System.IO.SearchOption.AllDirectories);
                foreach (String f in files)
                {
                    String lowerCaseFileName = f.ToLower();
                    if (lowerCaseFileName.EndsWith(".exe") || lowerCaseFileName.EndsWith(".dll"))
                        result.Add(f);
                }
            }
            return result;
        }

        private static int ShowHelpMenu()
        {
            StringBuilder sb = new StringBuilder("usage:\t\tscopegen.exe -target:<target> -sys:<sysfile> {-files:<files>|-paths:<paths>|-precomp:<compdir>} [-meta:<meta>] [-refs:<refs>]");
            sb.Append(System.Environment.NewLine);
            sb.Append(System.Environment.NewLine);
            sb.Append("?\t\tprint this help");
            sb.Append(System.Environment.NewLine);
            sb.Append("<target>\ttarget scope path");
            sb.Append(System.Environment.NewLine);
            sb.Append("<sysfile>\tsystem assembly path");
            sb.Append(System.Environment.NewLine);
            sb.Append("<files>\t\tcomma-separated list of application assembly files");
            sb.Append(System.Environment.NewLine);
            sb.Append("<paths>\t\tcomma-separated list of application assembly directories");
            sb.Append(System.Environment.NewLine);
            sb.Append("<compdir>\tsource directory of script files to precompile");
            sb.Append(System.Environment.NewLine);
            sb.Append("<meta>\t\ttarget meta-scope path (optional)");
            sb.Append(System.Environment.NewLine);
            sb.Append("<refs>\t\treference paths to be used during dependency lookup (optional)");
            Console.WriteLine(sb.ToString());
            return -1;
        }
    }
}
