#!/bin/bash

# ----------------------------------------------------------------------------
#  Static Analyzer script for Linux/Cygwin/OSX
# ----------------------------------------------------------------------------

# ----------------------------------------------------------------------------
# THIS PRODUCT CONTAINS RESTRICTED MATERIALS OF HCL
# AppScan on Cloud (C) COPYRIGHT HCL Technologies, 2016, 2025
# All Rights Reserved * Licensed Materials - Property of HCL
# US Government Users Restricted Rights - Use, duplication or disclosure
# restricted by GSA ADP Schedule Contract with HCL Technologies.
# ----------------------------------------------------------------------------

# ----------------------------------------------------------------------------
# ENVIRONMENT VARIABLES
#
# Optional:
# 	APPSCAN_OPTS - Parameters passed to the Java JVM
# 		e.g. To enable FIPS for HCL SDK use:
#			set APPSCAN_OPTS=-Dcom.ibm.jsse2.usefipsprovider=true property
# ----------------------------------------------------------------------------

if [ -n "$APPSCAN_INTERNAL" ] ; then
	APPSCAN_OPTS="-DINTERNAL_SERVER=$APPSCAN_INTERNAL $APPSCAN_OPTS"
	echo .
	echo WARNING: The APPSCAN_INTERNAL environment variable has been replaced with the Java property -DINTERNAL_SERVER.
	echo e.g. export APPSCAN_OPTS=-DINTERNAL_SERVER=\<server\>
	echo .
fi
if [ -n "$APPSCAN_DOMAIN" ] ; then
	APPSCAN_OPTS="-DBLUEMIX_SERVER=$APPSCAN_DOMAIN $APPSCAN_OPTS"
	echo .
	echo WARNING: The APPSCAN_DOMAIN environment variable has been replaced with the Java property -DBLUEMIX_SERVER.
	echo e.g. export APPSCAN_OPTS=-DBLUEMIX_SERVER=\<server\>
	echo .
fi

die ( ) {
    echo
    echo "$*"
    echo
    exit 1
}

# Create product temp folders if those doesn't exist
APPSCAN_WD=~/.appscan
if [ ! -d $APPSCAN_WD ] ; then
	mkdir $APPSCAN_WD
fi
if [ ! -w $APPSCAN_WD ] ; then
	die "ERROR: $APPSCAN_WD is not writable"
fi
		
if [ ! -d $APPSCAN_WD/temp ] ; then
	mkdir $APPSCAN_WD/temp 
fi
if [ ! -w $APPSCAN_WD/temp ] ; then
	die "ERROR: $APPSCAN_WD/temp is not writable"
fi

export APPSCAN_INSTALL_DIR=`dirname "${BASH_SOURCE[0]}"`/../
if [ "x$APPSCAN_INSTALL_DIR" == x ] ; then
	die "ERROR: APPSCAN_INSTALL_DIR is not set"
fi

# For Cygwin, ensure paths are in UNIX format before anything is touched.
if [[ `uname` == CYGWIN* ]] ; then
    [ -n "$JAVA_HOME" ] && JAVA_HOME=`cygpath --unix "$JAVA_HOME"`
fi

# Set APPSCAN_IRGEN_CLIENT if it doesn't exist
if [ ! -n "$APPSCAN_IRGEN_CLIENT" ] ; then
	export APPSCAN_IRGEN_CLIENT=CLI
fi

# Use Apple Silicon JRE if possible
if [ "$(uname -s)" = "Darwin" -a "$(uname -m)" = "arm64" -a -d $APPSCAN_INSTALL_DIR/jre-arm64 ]; then
    export JAVA_HOME=$APPSCAN_INSTALL_DIR/jre-arm64
fi

# Set JAVA_HOME if it doesn't exist or it's too old
if [[ ! -d "$JAVA_HOME" || $($JAVA_HOME/bin/java -version 2>&1) =~ 'version "1.8' ]]; then
	export JAVA_HOME=$APPSCAN_INSTALL_DIR/jre
fi

# Process additional configuration
if [ -f "$APPSCAN_INSTALL_DIR/config/cli.config" ] ; then
	CLI_CONFIG_OPTS="$(paste -s $APPSCAN_INSTALL_DIR/config/cli.config)"
fi

export LD_LIBRARY_PATH=$APPSCAN_INSTALL_DIR/bin:$LD_LIBRARY_PATH
export DYLD_LIBRARY_PATH=$APPSCAN_INSTALL_DIR/bin:$DYLD_LIBRARY_PATH

JAVACMD="$JAVA_HOME/bin/java"

if [ $# -eq 0 ] ; then
        "$JAVACMD" $CLI_CONFIG_OPTS $APPSCAN_OPTS -cp "$APPSCAN_INSTALL_DIR/lib/*" com.ibm.appscan.cli.common.Launcher "$APPSCAN_INSTALL_DIR" help
elif [ "$1" == "update" ] ; then

		umask 0022
        "$JAVACMD" --module-path "$APPSCAN_INSTALL_DIR/fips" --add-opens java.base/java.security=ALL-UNNAMED --add-exports java.base/sun.security.internal.spec=ALL-UNNAMED $CLI_CONFIG_OPTS $APPSCAN_OPTS -cp "$APPSCAN_INSTALL_DIR/lib/*" com.ibm.appscan.cli.common.Launcher "$APPSCAN_INSTALL_DIR" update

        if [ $? -ne 0 ] ; then
                die ""
        fi

		echo
		echo -e "Completing the update process..."
		
		# Get the absolute path of the original install and the directory name.
		TEMP_INSTALL_DIR=`dirname "${BASH_SOURCE[0]}"`
		ORIGINAL_INSTALL_DIR="$(dirname -- "$(cd "$TEMP_INSTALL_DIR" && pwd)")"

		# Handle macOS bundle structure
		BUNDLE_DIR="/Contents/Home"
		if [[ "$(uname -s)" == "Darwin" && "$ORIGINAL_INSTALL_DIR" == *"$BUNDLE_DIR" ]]; then
			ORIGINAL_INSTALL_DIR=${ORIGINAL_INSTALL_DIR%"$BUNDLE_DIR"}
		fi

		ORIGINAL_WORKING_DIR=`pwd`
		ORIGINAL_INSTALL_NAME="${ORIGINAL_INSTALL_DIR##*/}"
		PARENT_OF_INSTALL=$(dirname -- "$ORIGINAL_INSTALL_DIR")
		
		# If the original install directory name is of the form SAClientUtil.x.y.zzzz, use SAClientUtil instead.
		# Also adjust the original working directory name so we can reuse it.
		if [[ "$ORIGINAL_INSTALL_NAME" =~ ^SAClientUtil\.[0-9]\.[0-9]\.[0-9]+ ]] ; then
        	ORIGINAL_INSTALL_NAME="SAClientUtil"
        	ORIGINAL_WORKING_DIR=$(echo "$ORIGINAL_WORKING_DIR" | sed "s/SAClientUtil\.[0-9]\.[0-9]\.[0-9]+/SAClientUtil/")
		fi
		
		# If we're in the original install directory, cd out of it and remember we did.
		in_original_install=false
		if [[ `pwd` == *"$ORIGINAL_INSTALL_DIR"* ]] ; then
        	cd "$ORIGINAL_INSTALL_DIR"
        	in_original_install=true
        fi
        
        # Delete the original install directory.
		rm -rf "$ORIGINAL_INSTALL_DIR"

		# Move the new install to the original location.
		mv $APPSCAN_WD/temp/SAClientUtil "$PARENT_OF_INSTALL"
		
		# Rename it if necessary
		DESTINATION="$PARENT_OF_INSTALL/$ORIGINAL_INSTALL_NAME"
		if [ "$DESTINATION" != "$PARENT_OF_INSTALL/SAClientUtil" ] ; then
			mv "$PARENT_OF_INSTALL/SAClientUtil" "$DESTINATION"
		fi
		
		echo -e "Update complete.\nThe updated installation is located at:\n$DESTINATION"
		
		# Alert the user that they must cd to the new install location if they were running from the previous install directory.
		if [ "$in_original_install" = true ] ; then
    		echo -e "You must explicitly change directory (cd) to the new install directory to continue working in this shell."
		fi

else
	"$JAVACMD" -Xmx8g --module-path "$APPSCAN_INSTALL_DIR/fips" --add-opens java.base/java.security=ALL-UNNAMED --add-exports java.base/sun.security.internal.spec=ALL-UNNAMED $CLI_CONFIG_OPTS $APPSCAN_OPTS -cp "$APPSCAN_INSTALL_DIR/lib/*" com.ibm.appscan.cli.common.Launcher "$APPSCAN_INSTALL_DIR" "$@"
fi

echo
exit $?
