using System;
using System.Collections.Generic;
using System.Text;
using Common;

namespace CSharp
{
    [Serializable]
    public class Driver : Common.IDriver
    {
        #region Fields
        private int cacheClearCounter = 0;
        private readonly Object delegte;
        private readonly AppDomain domain;
        private readonly System.Collections.Generic.IDictionary<String, System.Reflection.MethodInfo> delegateMethods;
        #endregion

        #region Static Fields
        private static Driver driver;
        #endregion

        #region Constants
        private const int CACHE_CLEAR_CYCLE             = 50000;
        private const int SOFT_MAX_MEMORY_BUDGET        = 200;
        private const int HARD_MAX_MEMORY_BUDGET        = 250;
        public const String INVALID_SOURCE_FILE         = null;
        public const ulong INVALID_LINE_NUMBER          = ulong.MaxValue;
        private const int INVALID_HANDLE                = -1;
        private const Boolean DO_PROFILING              = true;
        private const Boolean ALLOW_CLEAR_CACHES        = true;
        private const Int32 PROFILES_DUMP_THRESHOLD     = 1000;
        private const String CSHARP_IMPL_STRONG_NAME    = "CSharpImpl, Version=1.0.0.0, Culture=neutral, PublicKeyToken=4e08b76868af6117";
        #endregion

        #region CTOR
		public Driver() : this(CSHARP_IMPL_STRONG_NAME) {

		}
	
		public Driver(String implName)
        {			
            domain = AppDomain.CreateDomain(System.Guid.NewGuid().ToString());
            domain.ExecuteAssemblyByName(implName);

			int firstSpace = implName.IndexOf(' ');
			String asmName = (firstSpace > 0)? implName.Substring(0, firstSpace-1) : implName;
			foreach (System.Reflection.Assembly asm in domain.GetAssemblies())
            {
                if (asm.FullName.StartsWith(asmName))
                {
                    System.Runtime.Remoting.ObjectHandle delegteHandle = domain.CreateInstance(asm.FullName, asmName + ".Driver");
                    delegte = delegteHandle.Unwrap();
                    break;
                }
            }
            System.Diagnostics.Debug.Assert(delegte != null);
            delegateMethods = new System.Collections.Generic.Dictionary<String, System.Reflection.MethodInfo>();
            foreach (System.Reflection.MethodInfo mi in delegte.GetType().GetMethods())
            {
                delegateMethods.Add(mi.Name, mi);
            }
        }
        #endregion

        #region DTOR
        ~Driver()
        {
            try
            {
                AppDomain.Unload(domain);
            }
            catch (CannotUnloadAppDomainException)
            { 
            }
        }
        #endregion

        #region Methods
        public static void InvalidateDriver()
        {
            driver = null;
        }

        public static Driver GetDriver()
        {
            if (driver == null)
                driver = new Driver();
            return driver;
        }

        private void ConditionalClearCaches()
        {
            if (ALLOW_CLEAR_CACHES)
            {
                int memoryUsageInMB = (int)(1 + (GC.GetTotalMemory(false) >> 20));
                if (memoryUsageInMB > SOFT_MAX_MEMORY_BUDGET)
                {
                    if ((memoryUsageInMB > HARD_MAX_MEMORY_BUDGET) || ((cacheClearCounter % CACHE_CLEAR_CYCLE) == 0))
                    {
                        delegateMethods["ClearCaches"].Invoke(delegte, null);
                        cacheClearCounter = 1;
                    }
                    else
                    {
                        cacheClearCounter++;
                    }
                }
            }
        }

        private void PerformVoidDelegation(String mname, params Object[] args)
        {
            ConditionalClearCaches();
            delegateMethods[mname].Invoke(delegte, args);
        }

        private T PerformDelegation<T>(String mname, params Object[] args)
        {
            ConditionalClearCaches();
            return (T) delegateMethods[mname].Invoke(delegte, args);
        }

        public void AddSourceMappings(int asmIndex, System.Collections.Generic.IDictionary<String, System.Collections.Generic.List<String>> mappings)
        {
            int[] allClasses = ImageAllClasses(asmIndex);
            foreach (int cls in allClasses)
            {
                String rawClsName = ClassGetName(cls);
                if (rawClsName != null)
                {
                    String clsName = rawClsName.Replace('/', '.').Replace('+', '.');
                    if (mappings.ContainsKey(clsName))
                    {
                        PerformVoidDelegation("RegisterClassSourceFiles", cls, mappings[clsName].ToArray());
                    }
                }
            }
        }

        public String GetPdbPath(int asmIndex)
        {
            return PerformDelegation<String>("GetPdbPath", asmIndex);
        }

        public String[] ClassGetSourceFiles(int clsIndex)
        {
            return PerformDelegation<String[]>("ClassGetSourceFiles", clsIndex);
        }

        public Int32 ClassGetStaticCtor(int clsIndex)
        {
            return PerformDelegation<Int32>("ClassGetStaticCtor", clsIndex);
        }

        public Int32 AttributeGetType(int attIndex)
        {
            return PerformDelegation<Int32>("AttributeGetType", attIndex);
        }

        public Int32 AttributeArgumentCount(int attIndex)
        {
            return PerformDelegation<Int32>("AttributeArgumentCount", attIndex);
        }

		public Int32 AttributeArgumentType(int attIndex, int argIndex)
        {
            return PerformDelegation<Int32>("AttributeArgumentType", attIndex, argIndex);
        }

		public string AttributeArgumentValue(int attIndex, int argIndex)
        {
            return PerformDelegation<string>("AttributeArgumentValue", attIndex, argIndex);
        }

		public Int32[] MethodGetCustomAttributes(int methodIndex)
        {
            return PerformDelegation<Int32[]>("MethodGetCustomAttributes", methodIndex);
        }

	public Int32[] MethodGetCustomParameterAttributes(int methodIndex, int parameterIndex)
	{
            return PerformDelegation<Int32[]>("MethodGetCustomParameterAttributes", methodIndex, parameterIndex);
	}
	
        public Int32[] ClassGetCustomAttributes(int clsIndex)
        {
            return PerformDelegation<Int32[]>("ClassGetCustomAttributes", clsIndex);
        }

        public Int32 FindImage(String imagePath)
        {
            return PerformDelegation<Int32>("FindImage", imagePath);
        }
		
		public Int32 GetImage(String shortName) {
	        return PerformDelegation<Int32>("GetImage", shortName);
		}
		
        public Int32 MethodGetMaxStackHeight(int index)
        {
            return PerformDelegation<Int32>("MethodGetMaxStackHeight", index);
        }

        public String ImageGetAssemblyName(int index)
        {
            return PerformDelegation<String>("ImageGetAssemblyName", index);
        }

        public String[] ImageGetRefrencedAssamblies(int asmIndex)
        {
            return PerformDelegation<String[]>("ImageGetRefrencedAssamblies", asmIndex);
        }

        public void AddLineMappings(int methodIndex, IDictionary<Range, LineInfo> mappings)
        {
            foreach (KeyValuePair<Range, LineInfo> e in mappings)
            {
                PerformVoidDelegation("RegisterLineMappings", methodIndex, e.Key.Start, e.Key.Length, e.Value.FileName, e.Value.LineNumber);
            }
        }

        public String MethodGetSourceFileName(int methodIndex, ulong offset)
        {
            return PerformDelegation<String>("MethodGetSourceFileName", methodIndex, offset);
        }

        public UInt64 MethodGetLineNumber(int methodIndex, ulong offset)
        {
            return PerformDelegation<UInt64>("MethodGetLineNumber", methodIndex, offset);
        }

        public Boolean MethodIsNative(int index)
        {
            return PerformDelegation<Boolean>("MethodIsNative", index);
        }

        public Int32 MethodGetMaxLocals(int index)
        {
            return PerformDelegation<Int32>("MethodGetMaxLocals", index);
        }

        public Int32 NumberOfClasses(int index)
        {
            return PerformDelegation<Int32>("NumberOfClasses", index);
        }

        public Int32[] ImageAllClasses(int asmIndex)
        {
            return PerformDelegation<Int32[]>("ImageAllClasses", asmIndex);
        }

        public Int32[] ClassGetDeclaredInterfaces(int clsIndex)
        {
            return PerformDelegation<Int32[]>("ClassGetDeclaredInterfaces", clsIndex);
        }

        public Int32[] ClassGetDeclaredStaticMethods(int clsIndex)
        {
            return PerformDelegation<Int32[]>("ClassGetDeclaredStaticMethods", clsIndex);
        }

        public Int32[] ClassGetDeclaredInstanceMethods(int clsIndex)
        {
            return PerformDelegation<Int32[]>("ClassGetDeclaredInstanceMethods", clsIndex);
        }

        public Boolean FieldIsPublic(int fieldIndex)
        {
            return PerformDelegation<Boolean>("FieldIsPublic", fieldIndex);
        }

        public Boolean FieldIsProtected(int fieldIndex)
        {
            return PerformDelegation<Boolean>("FieldIsProtected", fieldIndex);
        }

        public Boolean FieldIsFinal(int fieldIndex)
        {
            return PerformDelegation<Boolean>("FieldIsFinal", fieldIndex);
        }

        public Int32[] ClassGetDeclaredStaticFields(int clsIndex)
        {
            return PerformDelegation<Int32[]>("ClassGetDeclaredStaticFields", clsIndex);
        }

        public Int32[] ClassGetDeclaredInstanceFields(int clsIndex)
        {
            return PerformDelegation<Int32[]>("ClassGetDeclaredInstanceFields", clsIndex);
        }

        public Int32 ClassGetParent(int clsIndex)
        {
            return PerformDelegation<Int32>("ClassGetParent", clsIndex);
        }

        public Int32 ClassGetImage(int clsIndex)
        {
            return PerformDelegation<Int32>("ClassGetImage", clsIndex);
        }

        public Int32 FindClass(int asmIndex, String clsName, String pkgName)
        {
            return PerformDelegation<Int32>("FindClass", asmIndex, clsName, pkgName);
        }

        public Int32 MethodGetReturnType(int methodIndex)
        {
            return PerformDelegation<Int32>("MethodGetReturnType", methodIndex);
        }

        public Int32 MethodGetNumberOfParameters(int methodIndex)
        {
            return PerformDelegation<Int32>("MethodGetNumberOfParameters", methodIndex);
        }

        public String MethodGetName(int methodIndex)
        {
            return PerformDelegation<String>("MethodGetName", methodIndex);
        }

        public String ClassGetName(int clsIndex)
        {
            return PerformDelegation<String>("ClassGetName", clsIndex);
        }

        public Boolean ClassIsPublic(int clsIndex)
        {
            return PerformDelegation<Boolean>("ClassIsPublic", clsIndex);
        }

        public Boolean ClassIsPrimitive(int clsIndex)
        {
            return PerformDelegation<Boolean>("ClassIsPrimitive", clsIndex);
        }
        
        public Boolean ClassIsReference(int clsIndex)
        {
            return PerformDelegation<Boolean>("ClassIsReference", clsIndex);
        }

        public Boolean ClassIsWith(int clsIndex)
        {
            return PerformDelegation<Boolean>("ClassIsWith", clsIndex);
        }

        public Boolean ClassIsTypeVar(int clsIndex)
        {
            return PerformDelegation<Boolean>("ClassIsTypeVar", clsIndex);
        }

        public Boolean ClassIsAbstract(int clsIndex)
        {
            return PerformDelegation<Boolean>("ClassIsAbstract", clsIndex);
        }

        public Boolean ClassIsInterface(int clsIndex)
        {
            return PerformDelegation<Boolean>("ClassIsInterface", clsIndex);
        }

        public Boolean ClassIsArray(int clsIndex)
        {
            return PerformDelegation<Boolean>("ClassIsArray", clsIndex);
        }

        public Boolean ClassIsPointer(int clsIndex)
        {
            return PerformDelegation<Boolean>("ClassIsPointer", clsIndex);
        }

        public Byte[] MethodGetBytecode(int methodIndex)
        {
            return PerformDelegation<Byte[]>("MethodGetBytecode", methodIndex);
        }

        public Boolean MethodIsStatic(int methodIndex)
        {
            return PerformDelegation<Boolean>("MethodIsStatic", methodIndex);
        }

        public Boolean MethodIsPrivate(int methodIndex)
        {
            return PerformDelegation<Boolean>("MethodIsPrivate", methodIndex);
        }

        public Boolean MethodIsInit(int methodIndex)
        {
            return PerformDelegation<Boolean>("MethodIsInit", methodIndex);
        }

        public Boolean MethodIsFinal(int methodIndex)
        {
            return PerformDelegation<Boolean>("MethodIsFinal", methodIndex);
        }

        public Boolean MethodIsAbstract(int methodIndex)
        {
            return PerformDelegation<Boolean>("MethodIsAbstract", methodIndex);
        }

        public Boolean MethodIsPublic(int methodIndex)
        {
            return PerformDelegation<Boolean>("MethodIsPublic", methodIndex);
        }

        public Int32 ArrayGetElement(int arrayClsIndex)
        {
            return PerformDelegation<Int32>("ArrayGetElement", arrayClsIndex);
        }

        public Int32 MethodGetHeaderSize(int methodIndex)
        {
            return PerformDelegation<Int32>("MethodGetHeaderSize", methodIndex);
        }

        public ExceptionEntry[] MethodGetHandlers(int methodIndex)
        {
            return PerformDelegation<ExceptionEntry[]>("MethodGetHandlers", methodIndex);
        }

        public Int32 MethodGetParameterType(int methodIndex, int paramIndex)
        {
            return PerformDelegation<Int32>("MethodGetParameterType", methodIndex, paramIndex);
        }

        public String PrimitiveGetName(int index)
        {
            return PerformDelegation<String>("PrimitiveGetName", index);
        }

        public Int32 WithGetParam(int clsIndex, int paramIndex)
        {
            return PerformDelegation<Int32>("WithGetParam", clsIndex, paramIndex);
        }

        public Int32 RefGetReferrent(int clsIndex)
        {
            return PerformDelegation<Int32>("RefGetReferrent", clsIndex);
        }

        public String ResolveStringToken(int methodIndex, int token)
        {
            return PerformDelegation<String>("ResolveStringToken", methodIndex, token);
        }

        public Int32 WithGetNumParams(int clsIndex)
        {
            return PerformDelegation<Int32>("WithGetNumParams", clsIndex);
        }

        public Int32 MethodGetDeclaringClass(int methodIndex)
        {
            return PerformDelegation<Int32>("MethodGetDeclaringClass", methodIndex);
        }

        public Int32 ResolveMethodToken(int methodIndex, int token)
        {
            return PerformDelegation<Int32>("ResolveMethodToken", methodIndex, token);
        }

        public Int32 ResolveFieldToken(int methodIndex, int token)
        {
            return PerformDelegation<Int32>("ResolveFieldToken", methodIndex, token);
        }

        public Boolean FieldIsStatic(int fieldIndex)
        {
            return PerformDelegation<Boolean>("FieldIsStatic", fieldIndex);
        }

        public Int32 FieldGetType(int fieldIndex)
        {
            return PerformDelegation<Int32>("FieldGetType", fieldIndex);
        }

        public Int32 FieldDeclaringClass(int fieldIndex)
        {
            return PerformDelegation<Int32>("FieldDeclaringClass", fieldIndex);
        }

        public Boolean ClassIsValueType(int clsIndex)
        {
            return PerformDelegation<Boolean>("ClassIsValueType", clsIndex);
        }

        public Int32 MethodGetLocalVariableType(int methodIndex, int localIndex)
        {
            return PerformDelegation<Int32>("MethodGetLocalVariableType", methodIndex, localIndex);
        }

        public Int32 MethodGetAccessMask(int methodIndex)
        {
            return PerformDelegation<Int32>("MethodGetAccessMask", methodIndex);
        }

        public String FieldGetName(int fieldIndex)
        {
            return PerformDelegation<String>("FieldGetName", fieldIndex);
        }

        public Int32 ResolveTypeToken(int methodIndex, int token)
        {
            return PerformDelegation<Int32>("ResolveTypeToken", methodIndex, token);
        }

        public Int32 WithGetBase(int index)
        {
            return PerformDelegation<Int32>("WithGetBase", index);
        }

        public Int32 PointerGetReferrent(int clsIndex)
        {
            return PerformDelegation<Int32>("PointerGetReferrent", clsIndex);
        }

	    public Int32 MethodNumberOfOutParameters(int methodIndex)
        {
            return PerformDelegation<Int32>("MethodNumberOfOutParameters", methodIndex);
        }

        #endregion
    }
}
