using System;
using System.Collections.Generic;
using System.Text;
using System.Reflection.Emit;
using System.Reflection;

namespace CSharp
{
    public class ILReader : IEnumerable<ILInstruction>
    {
        private Byte[] byteArray;
        private Int32 position;
        private MethodBase enclosingMethod;

        private static OpCode[] oneByteOpCodes = new OpCode[0x100];
        private static OpCode[] twoByteOpCodes = new OpCode[0x100];
        
        static ILReader()
        {
            foreach (System.Reflection.FieldInfo fi in typeof(OpCodes).GetFields(BindingFlags.Public | BindingFlags.Static))
            {
                OpCode opCode = (OpCode) fi.GetValue(null);
                UInt16 value = (UInt16) opCode.Value;
                if (value < 0x100)
                {
                    oneByteOpCodes[value] = opCode;
                }
                else if ((value & 0xff00) == 0xfe00)
                {
                    twoByteOpCodes[value & 0xff] = opCode;
                }
            }
        }

        public ILReader(MethodBase enclosingMethod)
        {
            this.enclosingMethod    = enclosingMethod;
            MethodBody  body        = enclosingMethod.GetMethodBody();
            this.byteArray          = (body == null) ? new Byte[0] : body.GetILAsByteArray();
            this.position           = 0;
        }

        public override string ToString()
        {
            StringBuilder sb = new StringBuilder();
            sb.Append(".method ");
            sb.Append(enclosingMethod.IsPrivate ? "private " : "public ");
            sb.Append(enclosingMethod.IsHideBySig ? "hidebysig " : "");
            sb.Append(enclosingMethod.IsStatic ? "static " : "instance ");
            sb.Append(((System.Reflection.MethodInfo) enclosingMethod).ReturnType.Name);
            sb.Append(' ');
            sb.Append(enclosingMethod.Name);
            sb.Append('(');
            ParameterInfo[] parameters = enclosingMethod.GetParameters();
            for (int index = 0; index < parameters.Length; ++index)
            {
                sb.Append(parameters[index]);
                if (index < (parameters.Length - 1))
                {
                    sb.Append(',');
                }
            }
            sb.Append(')');
            sb.Append(' ');
            MethodImplAttributes implFlags = enclosingMethod.GetMethodImplementationFlags();
            if ((implFlags & MethodImplAttributes.IL) == MethodImplAttributes.IL)
            { 
                sb.Append("cil ");
            }
            if ((implFlags & MethodImplAttributes.Managed) == MethodImplAttributes.Managed)
            {
                sb.Append("managed ");
            }
            if ((implFlags & MethodImplAttributes.Native) == MethodImplAttributes.Native)
            {
                sb.Append("native ");
            }
            sb.Append(System.Environment.NewLine);
            sb.Append(".maxstack ");
            MethodBody methodBody = enclosingMethod.GetMethodBody();
            if (methodBody != null)
            {
                sb.Append(methodBody.MaxStackSize);
                sb.Append(System.Environment.NewLine);
                sb.Append(".locals ");
                foreach (System.Reflection.LocalVariableInfo localVarInfo in methodBody.LocalVariables)
                {
                    sb.Append(localVarInfo);
                    sb.Append(' ');
                }
                sb.Append(System.Environment.NewLine);
                sb.Append('{');
                sb.Append(System.Environment.NewLine);
                foreach (ILInstruction instr in this)
                {
                    sb.Append('\t');
                    sb.Append(instr);
                    sb.Append(System.Environment.NewLine);
                }
                sb.Append('}');
            }
            return sb.ToString();
        }

        public IEnumerator<ILInstruction> GetEnumerator()
        {
            while (position < byteArray.Length)
            {
                yield return Next();
            }
            position = 0;
            yield break;
        }
        
        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator() 
        { 
            return this.GetEnumerator(); 
        }

        public ILInstruction Next()
        {
            Int32 offset = position;
            OpCode opCode = OpCodes.Nop;
            Int32 token = 0;
     
            // Read first 1 or 2 bytes as 'opCode'.
            Byte code = ReadByte();
            if (code == 0xFE)
            {
                code = ReadByte();
                opCode = twoByteOpCodes[code];
            }
            else
            {
                opCode = oneByteOpCodes[code];
            }
     
            switch (opCode.OperandType)
            {
                case OperandType.InlineNone:
                    return new InlineNoneInstruction(enclosingMethod, opCode, offset);
                case OperandType.ShortInlineBrTarget:
                    SByte shortDelta = ReadSByte();
                    return new ShortInlineBrTargetInstruction(enclosingMethod, opCode, offset, shortDelta);
                case OperandType.InlineBrTarget: 
                    Int32 delta = ReadInt32();
                    return new InlineBrTargetInstruction(enclosingMethod, opCode, offset, delta);
                case OperandType.ShortInlineI:   
                    Byte int8 = ReadByte();
                    return new ShortInlineIInstruction(enclosingMethod, opCode, offset, int8);
                case OperandType.InlineI:        
                    Int32 int32 = ReadInt32();
                    return new InlineIInstruction(enclosingMethod, opCode, offset, int32);
                case OperandType.InlineI8:       
                    Int64 int64 = ReadInt64();
                    return new ShortInlineI8Instruction(enclosingMethod, opCode, offset, int64);
                case OperandType.ShortInlineR:   
                    Single float32 = ReadSingle();
                    return new ShortInlineRInstruction(enclosingMethod, opCode, offset, float32);
                case OperandType.InlineR:        
                    Double float64 = ReadDouble();
                    return new InlineRInstruction(enclosingMethod, opCode, offset, float64);
                case OperandType.ShortInlineVar: 
                    Byte index8 = ReadByte();
                    return new ShortInlineVarInstruction(enclosingMethod, opCode, offset, index8);
                case OperandType.InlineVar:      
                    UInt16 index16 = ReadUInt16();
                    return new InlineVarInstruction(enclosingMethod, opCode, offset, index16);
                case OperandType.InlineString:   
                    token = ReadInt32();
                    return new InlineStringInstruction(enclosingMethod, opCode, offset, token);
                case OperandType.InlineSig:      
                    token = ReadInt32();
                    return new InlineSigInstruction(enclosingMethod, opCode, offset, token);
                case OperandType.InlineField:    
                    token = ReadInt32();
                    return new InlineFieldInstruction(enclosingMethod, opCode, offset, token);
                case OperandType.InlineType:     
                    token = ReadInt32();
                    return new InlineTypeInstruction(enclosingMethod, opCode, offset, token);
                case OperandType.InlineTok:      
                    token = ReadInt32();
                    return new InlineTokInstruction(enclosingMethod, opCode, offset, token);
                case OperandType.InlineMethod:
                    token = ReadInt32();
                    return new InlineMethodInstruction(enclosingMethod, opCode, offset, token);
                case OperandType.InlineSwitch:
                    Int32 cases = ReadInt32();
                    Int32[] deltas = new Int32[cases];
                    for (Int32 i = 0; i < cases; i++)
                    {
                        deltas[i] = ReadInt32();
                    }
                    return new InlineSwitchInstruction(enclosingMethod, opCode, offset, deltas);
                default:
                    throw new BadImageFormatException("Unexpected operand type: " + opCode.OperandType + ".");
            }
        }
        
        Byte ReadByte() 
        { 
            return (Byte) byteArray[position++]; 
        }
        
        SByte ReadSByte() 
        { 
            return (SByte) ReadByte(); 
        }

        UInt16 ReadUInt16() 
        { 
            position += 2; 
            return BitConverter.ToUInt16(byteArray, position - 2); 
        }
            
        UInt32 ReadUInt32() 
        { 
            position += 4; 
            return BitConverter.ToUInt32(byteArray, position - 4); 
        }
            
        UInt64 ReadUInt64() 
        { 
            position += 8; 
            return BitConverter.ToUInt64(byteArray, position - 8); 
        }

        Int32 ReadInt32() 
        {   
            position += 4; 
            return BitConverter.ToInt32(byteArray, position - 4); 
        }
        
        Int64 ReadInt64() 
        { 
            position += 8; 
            return BitConverter.ToInt64(byteArray, position - 8); 
        }

        Single ReadSingle() 
        { 
            position += 4; 
            return BitConverter.ToSingle(byteArray, position - 4); 
        }
            
        Double ReadDouble() 
        { 
            position += 8; 
            return BitConverter.ToDouble(byteArray, position - 8); }
        } 
}