using System;
using System.Collections.Generic;
using System.Text;
using Common;

namespace CSharpImpl
{
    public class Driver : System.MarshalByRefObject, Common.IDriver
    {
        #region Fields
        protected readonly TwoWayMap<Int32, AssemblyInfo> assemblies;
        protected readonly TwoWayMap<String, AssemblyInfo> assembliesByName;
        private readonly TwoWayMap<String, AssemblyInfo> assembliesByShortName; // only uses name of assembly, no version or additional info
        private readonly TwoWayMap<Int32, ClassInfo> classes;
        private readonly TwoWayMap<Int32, FieldInfo> fields;
        protected readonly TwoWayMap<Int32, MethodInfo> methods;
        private readonly TwoWayMap<Int32, AttributeInfo> attributes;
        protected readonly System.Collections.Generic.IDictionary<Int32, System.Collections.Generic.IDictionary<Range, LineInfo>> lineMappings;
        private readonly System.Collections.Generic.Dictionary<Int32, String[]> sourceMappings;
        private readonly System.Collections.Generic.Dictionary<Int32, Int32> exceptionTokens;
        private readonly System.Collections.Generic.Dictionary<Int32, Int32[]> dependencies;
        private readonly System.Collections.Generic.Dictionary<Int32, Int32[]> staticMethodsCache;
        private readonly System.Collections.Generic.Dictionary<Int32, Int32> staticConstructorCache;
        private readonly System.Collections.Generic.Dictionary<Int32, Int32[]> instanceMethodsCache;
        private readonly System.Collections.Generic.Dictionary<Int32, Int32[]> classAttributesCache;
        private readonly System.Collections.Generic.Dictionary<Tuple<Int32,Int32>, Int32[]> methodParameterAttributesCache;
        private readonly System.Collections.Generic.Dictionary<Int32, Int32[]> methodAttributesCache;
        private readonly System.Collections.Generic.Dictionary<MethodTokenKey, Int32> methodTokensCache;
        private static System.TimeSpan elapsedTime;
        private static System.Collections.Generic.Dictionary<String, System.TimeSpan> timeProfiles;
        private static Int16 profileDumpCounter;
        private static CSharpImpl.Driver staticDriver = null;
        #endregion

        #region Constants
        public const String INVALID_SOURCE_FILE     = null;
        public const ulong INVALID_LINE_NUMBER      = ulong.MaxValue;
        private const int INVALID_HANDLE            = -1;
        private const Boolean DO_PROFILING          = false;
        private const Int32 PROFILES_DUMP_THRESHOLD = 1000;
        #endregion

        #region CTOR
        public Driver()
        {
            assemblies              = new TwoWayMap<Int32, AssemblyInfo>();
            assembliesByName        = new TwoWayMap<String, AssemblyInfo>();
            assembliesByShortName   = new TwoWayMap<String, AssemblyInfo>();
            classes                 = new TwoWayMap<Int32, ClassInfo>();
            methods                 = new TwoWayMap<Int32, MethodInfo>();
            attributes              = new TwoWayMap<Int32, AttributeInfo>();
            fields                  = new TwoWayMap<int, FieldInfo>();
            dependencies            = new Dictionary<Int32, Int32[]>();
            exceptionTokens         = new Dictionary<Int32, Int32>();
            instanceMethodsCache    = new Dictionary<Int32, Int32[]>();
            staticMethodsCache      = new Dictionary<Int32, Int32[]>();
            staticConstructorCache  = new Dictionary<Int32, Int32>();
            classAttributesCache    = new Dictionary<Int32, Int32[]>();
            methodAttributesCache   = new Dictionary<Int32, Int32[]>();
            methodParameterAttributesCache   = new Dictionary<Tuple<Int32,Int32>,Int32[]>();
            lineMappings            = new Dictionary<Int32, IDictionary<Range, LineInfo>>();
            sourceMappings          = new Dictionary<Int32, String[]>();
            methodTokensCache       = new Dictionary<MethodTokenKey, Int32>();
            // Set the static 'Driver' instance...
            if (staticDriver == null)
            {
                staticDriver        = this;                
            }
            else
            {
                throw new InvalidOperationException("Expected the static field 'staticDriver' to be null during construction time.");
            }
        }
        #endregion

        #region Static CTOR
        static Driver()
        {
            AppDomain.CurrentDomain.ReflectionOnlyAssemblyResolve += new ResolveEventHandler(CurrentDomain_ReflectionOnlyAssemblyResolve);
            if (DO_PROFILING)
            {
                timeProfiles = new Dictionary<string, TimeSpan>();
            }
        }
        #endregion

        #region Methods
        static System.Reflection.Assembly CurrentDomain_ReflectionOnlyAssemblyResolve(object sender, ResolveEventArgs args)
        {
            System.Reflection.Assembly refAsm = staticDriver.FindOrLoadImage(args.Name);
            return refAsm;
            
        }

        public override object InitializeLifetimeService()
        {
            return null;
        }

        private static void StartProfiling()
        {
            System.Diagnostics.Debug.Assert(DO_PROFILING);
            elapsedTime = new TimeSpan(System.DateTime.Now.Ticks);
        }

        private static void EndProfiling()
        {
            System.Diagnostics.Debug.Assert(DO_PROFILING);
            System.TimeSpan elapsed =
                new System.TimeSpan(System.DateTime.Now.Subtract(elapsedTime).Ticks);
            System.Diagnostics.StackTrace s = new System.Diagnostics.StackTrace();
            System.Diagnostics.StackFrame[] frames = s.GetFrames();
            System.Diagnostics.StackFrame callerFrame = frames[1];
            String callerName = callerFrame.GetMethod().Name;
            if (!timeProfiles.ContainsKey(callerName))
                timeProfiles.Add(callerName, new TimeSpan(0));
            timeProfiles[callerName] = timeProfiles[callerName].Add(elapsed);
            if (++profileDumpCounter == PROFILES_DUMP_THRESHOLD)
            {
                DumpProfiles();
                profileDumpCounter = 0;
            }
        }

        private static void DumpProfiles()
        {
            using (System.IO.TextWriter tw = new System.IO.StreamWriter("C:\\Temp\\TIMES.TXT", false))
            {
                foreach (String s in timeProfiles.Keys)
                {
                    StringBuilder sb = new StringBuilder(s);
                    sb.Append(": ");
                    sb.Append(timeProfiles[s].TotalMilliseconds);
                    sb.Append(" MS.");
                    tw.WriteLine(sb.ToString());
                }
            }
        }

        public void ClearCaches()
        {
            if (DO_PROFILING)
                StartProfiling();
            instanceMethodsCache.Clear();
            staticMethodsCache.Clear();
            classAttributesCache.Clear();
            methodAttributesCache.Clear();
            methodTokensCache.Clear();
            if (DO_PROFILING)
                EndProfiling();
        }

        public void RegisterLineMappings(int methodIndex, ulong offsetStart, ulong offsetLength, String fileName, ulong lineNumber)
        {
            if (DO_PROFILING)
                StartProfiling();
            Range r     = new Range(offsetStart, offsetLength);
            LineInfo li = new LineInfo(fileName, lineNumber);
            AddLineMappings(methodIndex, new KeyValuePair<Range, LineInfo>(r, li));
            if (DO_PROFILING)
                EndProfiling();
        }

        public void RegisterClassSourceFiles(int cls, String[] sourceFiles)
        {
            if (DO_PROFILING)
                StartProfiling();
            sourceMappings.Add(cls, sourceFiles);
            if (DO_PROFILING)
                EndProfiling();
        }

        private void AddLineMappings(int methodIndex, KeyValuePair<Range, LineInfo> e)
        {
            if (lineMappings.ContainsKey(methodIndex))
            {
                IDictionary<Range, LineInfo> m = lineMappings[methodIndex];
                if (!m.ContainsKey(e.Key))
                {
                    m.Add(e);
                }
            }
            else
            {
                IDictionary<Range, LineInfo> m = new Dictionary<Range, LineInfo>(1);
                m.Add(e);
                lineMappings.Add(methodIndex, m);
            }
        }

        //public void AddSourceMappings(int asmIndex, System.Collections.Generic.IDictionary<String, System.Collections.Generic.List<String>> mappings)
        //{
        //    //foreach (String s in mappings.Keys)
        //    //{
        //    //    Console.WriteLine(new StringBuilder("Reported mapping: ").Append(s).Append("<--->").Append(mappings[s]));
        //    //}
        //    int[] allClasses = ImageAllClasses(asmIndex);
        //    foreach (int cls in allClasses)
        //    {
        //        String rawClsName = ClassGetName(cls);
        //        //if (rawClsName == null)
        //        //{
        //        //    Console.WriteLine(new StringBuilder("ERROR: Could not find name of class at index ").Append(cls));
        //        //}
        //        if (rawClsName != null)
        //        {
        //            String clsName = rawClsName.Replace('/', '.').Replace('+', '.');
        //            if (mappings.ContainsKey(clsName))
        //            {
        //                sourceMappings.Add(cls, mappings[clsName].ToArray());
        //                //Console.WriteLine(new StringBuilder("Added source mappings for class at index ").Append(cls).Append(":").Append(clsName).Append("<--->").Append(mappings[clsName]));
        //            }
        //            else
        //            {
        //                //Console.WriteLine(new StringBuilder("ERROR: Could not find source mappings for ").Append(clsName));
        //            }
        //        }
        //    }
        //}

        public String GetPdbPath(int asmIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            AssemblyInfo asmInfo = assemblies[asmIndex];
            String asmLocation = asmInfo.Assembly.Location;
            StringBuilder sb = new StringBuilder(asmLocation.Remove(asmLocation.LastIndexOf('.')));
            sb.Append(".pdb");
            String path = sb.ToString();
            String result = (System.IO.File.Exists(path)) ? path : null;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public String[] ClassGetSourceFiles(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            String[] result;
            if (sourceMappings.ContainsKey(clsIndex))
            {
                result = sourceMappings[clsIndex];
            }
            else
            {
                result = null;
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 ClassGetStaticCtor(int clsIndex)
        {
			try {
            if (DO_PROFILING)
                StartProfiling();
            Int32 result;
            if (staticConstructorCache.ContainsKey(clsIndex))
            {
                result = staticConstructorCache[clsIndex];  
            }
            else 
            {
                ClassInfo clsInfo = classes[clsIndex];
                MethodInfo staticConstructor = clsInfo.StaticConstructor;
                if (staticConstructor != null)
                {
                    AddIfNeeded(staticConstructor);
                    staticConstructorCache.Add(clsIndex, methods[staticConstructor]);
                    result = staticConstructorCache[clsIndex];
                }
                else
                {
                    result = -1;
                }
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
			} catch (Exception e) {
				Console.WriteLine(e.StackTrace);
				throw e;
			}
        }

        public Int32 AttributeGetType(int attIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            ClassInfo c = AddIfNeeded(attributes[attIndex].Type);
            if (DO_PROFILING)
                EndProfiling();
            return classes[c];
        }
		
		public Int32 AttributeArgumentCount(int attIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            int count = attributes[attIndex].Arguments.GetLength(0);
            if (DO_PROFILING)
                EndProfiling();
            return count;
		}
		
		public Int32 AttributeArgumentType(int attIndex, int argIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            ClassInfo c = AddIfNeeded(attributes[attIndex].Arguments[argIndex].type);
            if (DO_PROFILING)
                EndProfiling();
            return classes[c];
		}
			
		public string AttributeArgumentValue(int attIndex, int argIndex)
        {
			try {
            if (DO_PROFILING)
                StartProfiling();
            string v = attributes[attIndex].Arguments[argIndex].val;
            if (DO_PROFILING)
                EndProfiling();
            return v;
			} catch (Exception e) {
				Console.WriteLine(e.StackTrace);
				throw e;
			}

        }

		public Int32[] MethodGetCustomParameterAttributes(int methodIndex, int parameterIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
	    Tuple<Int32,Int32> x = new Tuple<Int32,Int32>(methodIndex, parameterIndex);
            if (!methodParameterAttributesCache.ContainsKey(x))
            {
                MethodInfo methodInfo = methods[methodIndex];
		Vector<AttributeInfo> atts = methodInfo.GetParameterAnnotations (parameterIndex);
                Int32[] attIndices = new Int32[atts.Count];
                for (int index = 0; index < atts.Count; ++index)
                {
                    AddIfNeeded(atts[index]);
                    attIndices[index] = attributes[atts[index]];
                }
                methodParameterAttributesCache.Add(x, attIndices);
            }
            if (DO_PROFILING)
                EndProfiling();
            return methodParameterAttributesCache[x];
        }

        public Int32[] MethodGetCustomAttributes(int methodIndex)
        {
			try {
            if (DO_PROFILING)
                StartProfiling();
            if (!methodAttributesCache.ContainsKey(methodIndex))
            {
                MethodInfo methodInfo = methods[methodIndex];
                Vector<AttributeInfo> atts = methodInfo.Attributes;
                Int32[] attIndices = new Int32[atts.Count];
                for (int index = 0; index < atts.Count; ++index)
                {
                    AddIfNeeded(atts[index]);
                    attIndices[index] = attributes[atts[index]];
                }
                methodAttributesCache.Add(methodIndex, attIndices);
            }
            if (DO_PROFILING)
                EndProfiling();
            return methodAttributesCache[methodIndex];
			} catch (TypeLoadException e) {
				Console.WriteLine(e.StackTrace);
				throw e;
			}
        }

        public Int32[] ClassGetCustomAttributes(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            if (!classAttributesCache.ContainsKey(clsIndex))
            {
                ClassInfo clsInfo = classes[clsIndex];
                Vector<AttributeInfo> atts = clsInfo.Attributes;
                Int32[] attIndices = new Int32[atts.Count];
                for (int index = 0; index < atts.Count; ++index)
                {
                    AddIfNeeded(atts[index]);
                    attIndices[index] = attributes[atts[index]];
                }
                classAttributesCache.Add(clsIndex, attIndices);
            }
            if (DO_PROFILING)
                EndProfiling();
            return classAttributesCache[clsIndex];
        }

        public System.Reflection.Assembly FindOrLoadImage(String imagePath)
        {
            System.Reflection.Assembly refAsm = null;
            if (assembliesByName.ContainsKey(imagePath))
            {   // Use already loaded image
                refAsm = assembliesByName[imagePath].Assembly;
            }
            else
            {
                try
                {   // Attempt to load it
                    refAsm = System.Reflection.Assembly.ReflectionOnlyLoad(imagePath);
                }
                catch (System.Exception e)
                {   // Failed to load, see if we have a similiar (but different version) already loaded
                    String shortName = getShortName(imagePath);
                    if (shortName != null) {
						if (assembliesByShortName.ContainsKey(shortName)) {
                        	refAsm = assembliesByShortName[shortName].Assembly;
							Console.WriteLine("using " + refAsm.FullName + " for " + imagePath);
                    	} else {
							try {
	    	               		refAsm = System.Reflection.Assembly.ReflectionOnlyLoad(shortName);
								Console.WriteLine("loaded " + refAsm.FullName + " for " + imagePath);
							} catch (System.Exception ee) {
								e = ee;
							}
						}		
					}
					
                    if (refAsm == null)
                    {   // Can't find image, so rethrow exception
                        throw e;
                    }
                }
            }

			return refAsm;
        }

        public Int32 FindImage(String imagePath)
        {
			try {
            if (DO_PROFILING)
                StartProfiling();
            System.Reflection.Assembly resolvedAsm = null;
            try
            {
                resolvedAsm = System.Reflection.Assembly.ReflectionOnlyLoadFrom(imagePath);
            }
            catch (System.IO.FileLoadException)
            {
                String fileName = System.IO.Path.GetFileName(imagePath);
                foreach (System.Reflection.Assembly asm in AppDomain.CurrentDomain.GetAssemblies())
                {
                    String asmFileName = System.IO.Path.GetFileName(asm.Location);
                    if (asmFileName.Equals(fileName))
                    {
                        resolvedAsm = asm;
                        break;
                    }
                }
            }
            catch (System.IO.FileNotFoundException fnfE)
            {
                String shortName = getShortName(imagePath);
                if (shortName != null)
                {   // Try with the shorter non-versioned name
                    return FindImage(shortName);
                }
            }
            if (resolvedAsm != null)
            {
                Int32 result = FindImage(resolvedAsm);

                if (DO_PROFILING)
                    EndProfiling();
                return result;
            }
            else
            {
          ;      if (DO_PROFILING)
                    EndProfiling();
                throw new InvalidOperationException("FindImage Failed: " + imagePath);
                return 1;
            }
			} catch (Exception e) {
				Console.WriteLine(e.StackTrace);
				throw e;
			}
        }

        private Int32 FindImage(System.Reflection.Assembly asm)
        {
            AssemblyInfo asmInfo = null;
            foreach (String asmFullName in assembliesByName.GetKeys<String>())
            {
                if (asmFullName.Equals(asm.FullName))
                {
                    asmInfo = assembliesByName[asmFullName];
                    break;
                }
            }
            if (asmInfo == null)
            {
                String shortName = getShortName(asm.FullName);
                if (shortName != null)
                {   // Try finding it based on its Short Name only
                    foreach (String asmShortName in assembliesByShortName.GetKeys<String>())
                    {
                        if (asmShortName.Equals(shortName))
                        {
                            asmInfo = assembliesByShortName[asmShortName];
                            break;
                        }
                    }
                }
            }
            if (asmInfo == null)
            {
                asmInfo = new AssemblyInfo(asm);
            }
            if (!assemblies.ContainsKey(asmInfo))
            {
                assemblies.Add(assemblies.Count, asmInfo);
                assembliesByName.Add(asmInfo.Assembly.GetName().FullName, asmInfo);
                String shortName = getShortName(asmInfo.Assembly.GetName().FullName);
                if ((shortName != null) & !assembliesByShortName.ContainsKey(shortName))
                {
                    assembliesByShortName.Add(shortName, asmInfo);
                }
            }
            return assemblies[asmInfo];
        }
		
		public Int32 GetImage(String assemblyName) {
			for(int i = 0; i < assemblies.Count; i++) {
				if (assemblyName.Equals(ImageGetAssemblyName(i)) ){
					return i;
				}
			}
			return -1;
		}
			
        public Int32 MethodGetMaxStackHeight(int index)
        {
            if (DO_PROFILING)
                StartProfiling();
            Int32 result = methods[index].MaxStackHeight;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public String ImageGetAssemblyName(int index)
        {
            if (DO_PROFILING)
                StartProfiling();
            String result;
            if (assemblies.ContainsKey(index))
                result = assemblies[index].Name;
            else
                result = null;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

		public Int32 ImageGetNumberOfClasses(int index) {
            if (DO_PROFILING)
                StartProfiling();
            Int32 result;
            if (assemblies.ContainsKey(index))
                result = assemblies[index].ClassCount;
            else
                result = -1;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public String[] ImageGetRefrencedAssamblies(int asmIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            String[] res;
            if (assemblies.ContainsKey(asmIndex))
                res = assemblies[asmIndex].GetRefrencedAssamblies();
            else
                res = null;
            if (DO_PROFILING)
                EndProfiling();
            return res;
        }


        public String MethodGetSourceFileName(int methodIndex, ulong offset)
        {
            if (DO_PROFILING)
                StartProfiling();
            String result = INVALID_SOURCE_FILE;
            if (lineMappings.ContainsKey(methodIndex))
            {
                IDictionary<Range, LineInfo> image = lineMappings[methodIndex];
                foreach (Range r in image.Keys)
                {
                    if (r.Contains(offset))
                    {
                        result = image[r].FileName;
                        break;
                    }
                }
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public virtual UInt64 MethodGetLineNumber(int methodIndex, ulong offset)
        {
            if (DO_PROFILING)
                StartProfiling();
            UInt64 result = INVALID_LINE_NUMBER;
            if (lineMappings.ContainsKey(methodIndex))
            {
                IDictionary<Range, LineInfo> image = lineMappings[methodIndex];
                foreach (Range r in image.Keys)
                {
                    if (r.Contains(offset))
                    {
                        result = image[r].LineNumber;
                        break;
                    }
                }
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean MethodIsNative(int index)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = methods[index].IsNative;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }
		
		public Int32 MethodNumberOfOutParameters(int index) {
           if (DO_PROFILING)
                StartProfiling();
            Int32 result = methods[index].NumberOfOutParameters;
            if (DO_PROFILING)
                EndProfiling();
            return result;				
		}
		
        public Int32 MethodGetMaxLocals(int index)
        {
            if (DO_PROFILING)
                StartProfiling();
            Int32 result = methods[index].LocalVariablesCount;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 NumberOfClasses(int index)
        {
            if (DO_PROFILING)
                StartProfiling();
            Int32 result = assemblies[index].ClassCount;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32[] ImageAllClasses(int asmIndex)
        {
			try {
            if (DO_PROFILING)
                StartProfiling();
            AssemblyInfo asmInfo = assemblies[asmIndex];
            Vector<ClassInfo> clsInfos = asmInfo.Classes;
            List<int> result = new List<int>(clsInfos.Count);
            for (int index = 0; index < clsInfos.Count; ++index)
            {
                if (clsInfos[index].Type != null)
                {
                    ClassInfo c = AddIfNeeded(clsInfos[index]);
                    result.Add(classes[c]);
                }
            }
            if (DO_PROFILING)
                EndProfiling();
            return result.ToArray();
			} catch (System.IO.FileNotFoundException e) {
				Console.WriteLine(e.StackTrace);
				throw e;
			}
        }

        public Int32[] ClassGetDeclaredInterfaces(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            ClassInfo clsInfo = classes[clsIndex];
            Vector<ClassInfo> interfaces = clsInfo.Interfaces;
            int[] result = new int[interfaces.Count];
            for (int index = 0; index < interfaces.Count; ++index)
            {
                result[index] = GetClassIndex(interfaces[index]);
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32[] ClassGetDeclaredStaticMethods(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            if (!staticMethodsCache.ContainsKey(clsIndex))
            {
                ClassInfo clsInfo = classes[clsIndex];
                Vector<MethodInfo> staticMethods = clsInfo.StaticMethods;
                int[] s = new int[staticMethods.Count];
                for (int index = 0; index < staticMethods.Count; ++index)
                {
                    if (!methods.ContainsKey(staticMethods[index]))
                    {
                        methods.Add(methods.Count, staticMethods[index]);
                    }
                    s[index] = methods[staticMethods[index]];
                }
                staticMethodsCache.Add(clsIndex, s);
            }
            if (DO_PROFILING)
                EndProfiling();
            return staticMethodsCache[clsIndex];
        }

        public Int32[] ClassGetDeclaredInstanceMethods(int clsIndex)
        {
			try {
            if (DO_PROFILING)
                StartProfiling();
            if (!instanceMethodsCache.ContainsKey(clsIndex))
            {
                ClassInfo clsInfo = classes[clsIndex];
                Vector<MethodInfo> instanceMethods = clsInfo.InstanceMethods;
                int[] i = new int[instanceMethods.Count];
                for (int index = 0; index < instanceMethods.Count; ++index)
                {
                    AddIfNeeded(instanceMethods[index]);
                    i[index] = methods[instanceMethods[index]];
                }
                instanceMethodsCache.Add(clsIndex, i);
            }
            if (DO_PROFILING)
                EndProfiling();
            return instanceMethodsCache[clsIndex];
			} catch (MissingMethodException e) {
				Console.WriteLine(e.StackTrace);
				Console.WriteLine(classes[clsIndex].Name + " - " + classes[clsIndex].Type.AssemblyQualifiedName);
				throw e;
			}
        }

        public bool FieldIsPublic(int fieldIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            bool result = fields[fieldIndex].IsPublic;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public bool FieldIsProtected(int fieldIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            FieldInfo fi    = fields[fieldIndex];
            bool result     = ((!fi.IsPublic) && (!fi.IsPrivate));
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public bool FieldIsFinal(int fieldIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            bool result = fields[fieldIndex].IsInitOnly;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32[] ClassGetDeclaredStaticFields(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            ClassInfo clsInfo = classes[clsIndex];
            Vector<FieldInfo> staticFields = clsInfo.StaticFields;
            int[] result = new int[staticFields.Count];
            for (int index = 0; index < staticFields.Count; ++index)
            {
                result[index] = GetFieldIndex(staticFields[index]);
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32[] ClassGetDeclaredInstanceFields(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            ClassInfo clsInfo = classes[clsIndex];
            Vector<FieldInfo> instanceFields = clsInfo.InstanceFields;
            int[] result = new int[instanceFields.Count];
            for (int index = 0; index < instanceFields.Count; ++index)
            {
                result[index] = GetFieldIndex(instanceFields[index]);
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        private Int32 InvalidHandle(String msg)
        {
            //Console.WriteLine("INVALID HANDLE: " + msg);
            return INVALID_HANDLE;
        }

        public Int32 ClassGetParent(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            int result = (classes[clsIndex].Parent == null) ? /*InvalidHandle(new StringBuilder("ClassGetParent(").Append(clsIndex).Append(")").ToString())*/INVALID_HANDLE : GetClassIndex(classes[clsIndex].Parent);
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 ClassGetImage(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Int32 result;
            if (classes.ContainsKey(clsIndex))
            {
                ClassInfo clsInfo = classes[clsIndex];
                AssemblyInfo asmInfo = new AssemblyInfo(clsInfo.Type.Assembly);
                if (!assemblies.ContainsKey(asmInfo))
                {
                    String shortName = getShortName(asmInfo.Assembly.GetName().FullName);
                    if (shortName != null) 
                    {
                        if (assembliesByShortName.ContainsKey(shortName))
                        {   // Already have a similar assembly, so lets switch to it
                            asmInfo = assembliesByShortName[shortName];
                        }
                        else
                        {   
                            assemblies.Add(assemblies.Count, asmInfo);
                            assembliesByName.Add(asmInfo.Assembly.GetName().FullName, asmInfo);
                            assembliesByShortName.Add(shortName, asmInfo);
                        }
                    }
                }
                result = assemblies[asmInfo];
            }
            else
            {
                result = InvalidHandle(new StringBuilder("ClassGetImage(").Append(clsIndex).Append(")").ToString());
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        //private static Int32 FindClassByName(int asmIndex, String clsName)
        //{ 
        //    AssemblyInfo asmInfo            = assemblies[asmIndex];
        //    System.Reflection.Assembly asm  = asmInfo.Assembly;
        //    foreach (Type t in asm.GetTypes())
        //    {
        //        if (t.Name.EndsWith(clsName))
        //        {
        //            ClassInfo ci = new ClassInfo(t);
        //            if (classes.ContainsKey(ci))
        //            {
        //                return classes[ci];
        //            }
        //        }
        //    }
        //    return InvalidHandle(new StringBuilder("FindClassByName(").Append(asmIndex).Append(",").Append(clsName).Append(")").ToString());
        //}

        public Int32 FindClass(int asmIndex, String clsName, String pkgName)
        {
            Int32 result;
            if (DO_PROFILING)
                StartProfiling();
            int clsIndex = FindClassImpl(asmIndex, clsName, pkgName, new System.Collections.Generic.List<Int32>());
            if (clsIndex == INVALID_HANDLE)
            {
                result = InvalidHandle("Could not find class: " + clsName + " under package " + pkgName + " relative to assembly # " + asmIndex + ".");
            }
            else
            {
                result = clsIndex;
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        private Int32 FindClassImpl(int asmIndex, String clsName, String pkgName, System.Collections.Generic.List<Int32> visited)
        {
          try
          {
            return FindClassImplSub(asmIndex, clsName, pkgName, visited);
          }
          catch (Exception e)
          {
            return INVALID_HANDLE;
          }
        }

        private Int32 FindClassImplSub(int asmIndex, String clsName, String pkgName, System.Collections.Generic.List<Int32> visited)
        {
            visited.Add(asmIndex);
            System.Reflection.Assembly asm = assemblies[asmIndex].Assembly;

            // check if in this assembly
            String fullName;
            if ((pkgName != null) && (pkgName.Length > 0))
                fullName = new StringBuilder(pkgName).Append('.').Append(clsName).Replace('/', '.').ToString();
            else
                fullName = clsName.Replace('/', '.').ToString();

            Type match = asm.GetType(fullName);
		
            if (match != null)
            {
                ClassInfo key = AddIfNeeded(new ClassInfo(match));
                return classes[key];
            }

            // Recurse into dependent assemblies
            int[] dependencies = GetDependencies(asmIndex);
            foreach (int refAsmIndex in dependencies)
            {
                if (!visited.Contains(refAsmIndex))
                {
                    int result = FindClassImpl(refAsmIndex, clsName, pkgName, visited);
                    if (result != INVALID_HANDLE)
                        return result;
                }
            }

            //return InvalidHandle("Could not find class: " + clsName + " under package " + pkgName + " relative to assembly # " + asmIndex + ".");
            return INVALID_HANDLE;
        }

        private int[] GetDependencies(int asmIndex)
        {
            if (!dependencies.ContainsKey(asmIndex))
            {
                System.Reflection.Assembly asm = assemblies[asmIndex].Assembly;
                System.Collections.Generic.List<Int32> image = new List<int>();
                foreach (System.Reflection.AssemblyName refAsmName in asm.GetReferencedAssemblies())
                {
                    /* 
                     * We first try to resolve based on the assemblies that are already loaded. Otherwise
                     * fusion gets a stomach ache failing to resolve application assemblies, since the default binding
                     * is the directory from which the present DLL is executed.
                     */
                    System.Reflection.Assembly refAsm = FindOrLoadImage(refAsmName.FullName);
                    int refAsmIndex = FindImage(refAsm);
                    image.Add(refAsmIndex);
                }
                dependencies.Add(asmIndex, image.ToArray());
            }
            return dependencies[asmIndex];
        }

        private void AddIfNeeded(MethodInfo mi)
        {
            AddIfNeeded(mi.DeclaringClass);
            if (!methods.ContainsKey(mi))
            {
                methods.Add(methods.Count, mi);
            }
        }

        private void AddIfNeeded(AttributeInfo ai)
        {
            if (!attributes.ContainsKey(ai))
            {
                attributes.Add(ai, attributes.Count);
            }
        }

        private void AddIfNeeded(FieldInfo fi)
        {
            AddIfNeeded(fi.DeclaringClass);
            if (!fields.ContainsKey(fi))
            {
                fields.Add(fi, fields.Count);
            }
        }

        private ClassInfo AddIfNeeded(ClassInfo clsInfo)
        {
            if (!classes.ContainsKey(clsInfo))
            {
                AssemblyInfo asmInfo = new AssemblyInfo(clsInfo.Type.Assembly);
                ClassInfo foundClass = null;
                if (!assemblies.ContainsKey(asmInfo))
                {
                    String shortName = getShortName(asmInfo.Assembly.GetName().FullName);
                    if (shortName != null) 
                    {
                        if (assembliesByShortName.ContainsKey(shortName))
                        {   // Already have a similar assembly, so lets switch to it
                            asmInfo = assembliesByShortName[shortName];

                            String className = clsInfo.Name;
                            String packageName = "";
                            int iSlash = clsInfo.Name.LastIndexOf('/');
                            if (iSlash >= 0)
                            {
                                className = clsInfo.Name.Substring(iSlash + 1);
                                packageName = clsInfo.Name.Substring(0, iSlash);
                            }
                            Int32 hClass = FindClass(assemblies[asmInfo], className, packageName);
                            if (hClass != INVALID_HANDLE)
                            {
                                foundClass = classes[hClass];
                            }
                        }
                        else
                        {
                            assemblies.Add(assemblies.Count, asmInfo);
                            assembliesByName.Add(asmInfo.Assembly.GetName().FullName, asmInfo);
                            assembliesByShortName.Add(shortName, asmInfo);
                        }
                    }
                }
                if (foundClass == null)
                {
                    classes.Add(classes.Count, clsInfo);
                }
                else
                {
                    clsInfo = foundClass;
                }
            }
            return clsInfo;
        }

        private Int32 GetClassIndex(ClassInfo clsInfo)
        {
            ClassInfo c = AddIfNeeded(clsInfo);
            return classes[c];
        }

        private Int32 GetFieldIndex(FieldInfo fieldInfo)
        {
            AddIfNeeded(fieldInfo);
            return fields[fieldInfo];
        }

        public Int32 MethodGetReturnType(int methodIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            ClassInfo returnType = AddIfNeeded(methods[methodIndex].ReturnType);
            Int32 result = classes[returnType];
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 MethodGetNumberOfParameters(int methodIndex)
        {
			try {
            if (DO_PROFILING)
                StartProfiling();
            Int32 result = methods[methodIndex].NumberOfParameters;
            if (DO_PROFILING)
                EndProfiling();
            return result;
			} catch (Exception e) {
				Console.WriteLine(e.StackTrace);
				throw e;
			}
        }

        public String MethodGetName(int methodIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            String result = methods[methodIndex].Name;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public String ClassGetName(int clsIndex)
        {
            String result;
            if (DO_PROFILING)
                StartProfiling();
            if (classes.ContainsKey(clsIndex))
                result = classes[clsIndex].Name;
            else
                result = null;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean ClassIsPublic(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = classes[clsIndex].IsPublic;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean ClassIsPrimitive(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = classes[clsIndex].IsPrimitive;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean ClassIsReference(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = classes[clsIndex].IsByRef;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean ClassIsWith(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = classes[clsIndex].IsGeneric;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean ClassIsTypeVar(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = classes[clsIndex].IsGenericParameter;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean ClassIsAbstract(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = classes[clsIndex].IsAbstract;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean ClassIsInterface(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = classes[clsIndex].IsInterface;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean ClassIsArray(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = classes[clsIndex].IsArray;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean ClassIsPointer(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = classes[clsIndex].IsPointer;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public byte[] MethodGetBytecode(int methodIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            System.Reflection.MethodBase mi = methods[methodIndex].Method;
            System.Reflection.MethodBody body = mi.GetMethodBody();
            byte[] result = (body == null) ? null : body.GetILAsByteArray();
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean MethodIsStatic(int methodIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = methods[methodIndex].IsStatic;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean MethodIsPrivate(int methodIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = methods[methodIndex].IsPrivate;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean MethodIsInit(int methodIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = methods[methodIndex].IsInit;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        //public static Boolean MethodIsNative(int methodIndex)
        //{
        //    return methods[methodIndex].IsNative;
        //}

        public Boolean MethodIsFinal(int methodIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = methods[methodIndex].IsFinal;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean MethodIsAbstract(int methodIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = methods[methodIndex].IsAbstract;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean MethodIsPublic(int methodIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = methods[methodIndex].IsPublic;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 MethodGetAccessMask(int methodIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            int result = methods[methodIndex].GetAccessMask();
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 ArrayGetElement(int arrayClsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Int32 result = GetElementType(arrayClsIndex);
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 MethodGetHeaderSize(int methodIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Int32 result = methods[methodIndex].HeaderSize;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public ExceptionEntry[] MethodGetHandlers(int methodIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            IList<System.Reflection.ExceptionHandlingClause> clauses = methods[methodIndex].ExceptionEntries;
            ExceptionEntry[] result = new ExceptionEntry[clauses.Count];
            for (int index = 0; index < result.Length; ++index)
            {
                System.Reflection.ExceptionHandlingClause clause = clauses[index];
                String handlerKind;
                int catchTypeToken = INVALID_HANDLE;
                int filterOffset = -1;
                if ((clause.Flags & System.Reflection.ExceptionHandlingClauseOptions.Filter) == System.Reflection.ExceptionHandlingClauseOptions.Filter)
                {
                    handlerKind = "FILTER";
                    filterOffset = clause.FilterOffset;
                }
                else if ((clause.Flags & System.Reflection.ExceptionHandlingClauseOptions.Finally) == System.Reflection.ExceptionHandlingClauseOptions.Finally)
                {
                    handlerKind = "FINALLY";
                }
                else if ((clause.Flags & System.Reflection.ExceptionHandlingClauseOptions.Fault) == System.Reflection.ExceptionHandlingClauseOptions.Fault)
                {
                    handlerKind = "FAULT";
                }
                else
                {
                    handlerKind = "CATCH";
                    catchTypeToken = clause.CatchType.MetadataToken;
                    ClassInfo ci = AddIfNeeded(new ClassInfo(clause.CatchType));
                    if (!exceptionTokens.ContainsKey(catchTypeToken))
                        exceptionTokens.Add(catchTypeToken, classes[ci]);
                }
                result[index] = new ExceptionEntry(handlerKind, catchTypeToken, clause.TryOffset, clause.TryLength, filterOffset, clause.HandlerOffset, clause.HandlerLength);
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 MethodGetParameterType(int methodIndex, int paramIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            ClassInfo paramType = methods[methodIndex].GetParameterType(paramIndex);
            if (!classes.ContainsKey(paramType))
            {
                classes.Add(classes.Count, paramType);
            }
            Int32 result = classes[paramType];
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public String PrimitiveGetName(int index)
        {
            if (DO_PROFILING)
                StartProfiling();
            String fullName = classes[index].Name;
            //String result = fullName.Substring(fullName.LastIndexOf('/') + 1);
            if (DO_PROFILING)
                EndProfiling();
            return fullName;
        }

        public Int32 WithGetParam(int clsIndex, int paramIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Type paramType = classes[clsIndex].Type.GetGenericArguments()[paramIndex];
            ClassInfo ci = AddIfNeeded(new ClassInfo(paramType));
            Int32 result = classes[ci];
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 RefGetReferrent(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Int32 result = GetElementType(clsIndex);
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public String ResolveStringToken(int methodIndex, int token)
        {
            if (DO_PROFILING)
                StartProfiling();
            MethodInfo mi = methods[methodIndex];
            String result = mi.Method.Module.ResolveString(token);
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 WithGetNumParams(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Int32 result = classes[clsIndex].Type.GetGenericArguments().Length;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 MethodGetDeclaringClass(int methodIndex)
        {
            if (DO_PROFILING)
                StartProfiling();

            ClassInfo clsInfo = AddIfNeeded(new ClassInfo(methods[methodIndex].Method.DeclaringType));
            Int32 result = classes[clsInfo];
            
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        private System.Reflection.FieldInfo ResolveFieldTokenOnFail1(int methodIndex, int token)
        {
            MethodInfo mi = methods[methodIndex];
            System.Reflection.FieldInfo fi = null;
            try
            {
                fi = mi.Method.Module.ResolveField(token, mi.Method.DeclaringType.GetGenericArguments(), new Type[0]);
            }
            catch (System.NotSupportedException)
            {
            }
            catch (System.BadImageFormatException)
            {
            }
            catch (System.ArgumentException)
            {
            }
            return fi;
        }

        private System.Reflection.FieldInfo ResolveFieldTokenOnFail2(int methodIndex, int token)
        {
            MethodInfo mi = methods[methodIndex];
            System.Reflection.FieldInfo fi = null;
            try
            {
                fi = mi.Method.Module.ResolveField(token);
            }
            catch (System.NotSupportedException)
            {
            }
            catch (System.BadImageFormatException)
            {
            }
            catch (System.ArgumentException)
            {
            }
            return fi;
        }

        private System.Reflection.MethodBase ResolveMethodTokenOnFail1(int methodIndex, int token)
        {
            MethodInfo mi = methods[methodIndex];
            System.Reflection.MethodBase mb = null;
            try
            {
                mb = mi.Method.Module.ResolveMethod(token, mi.Method.DeclaringType.GetGenericArguments(), new Type[0]);
            }
            catch (System.NotSupportedException)
            {
            }
            catch (System.BadImageFormatException)
            {
            }
            catch (System.ArgumentException)
            {
            }
            return mb;
        }

        private System.Reflection.MethodBase ResolveMethodTokenOnFail2(int methodIndex, int token)
        {
            MethodInfo mi = methods[methodIndex];
            System.Reflection.MethodBase mb = null;
            try
            {
                mb = mi.Method.Module.ResolveMethod(token);
            }
            catch (System.NotSupportedException)
            {
            }
            catch (System.BadImageFormatException)
            {
            }
            catch (System.ArgumentException)
            {
            }
            return mb;
        }

        private sealed class MethodTokenKey
        {
            private readonly int methodIndex;
            private readonly int token;

            public MethodTokenKey(int methodIndex, int token)
            {
                this.methodIndex = methodIndex;
                this.token = token;
            }

            public override int GetHashCode()
            {
                return (token * methodIndex);
            }

            public override bool Equals(object obj)
            {
                if (obj is MethodTokenKey)
                {
                    MethodTokenKey k = (MethodTokenKey)obj;
                    if (k.methodIndex == this.methodIndex && k.token == this.token)
                        return true;
                }
                return false;
            }
        }

        public Int32 ResolveMethodToken(int methodIndex, int token)
        {
            if (DO_PROFILING)
                StartProfiling();
            Int32 result;
            MethodTokenKey key = new MethodTokenKey(methodIndex, token);
            if (methodTokensCache.ContainsKey(key))
            {
                result = methodTokensCache[key];
            }
            else
            {
                MethodInfo mi = methods[methodIndex];
                System.Reflection.MethodBase mb = null;
                try
                {
                    mb = mi.Method.Module.ResolveMethod(token, mi.Method.DeclaringType.GetGenericArguments(), mi.Method.GetGenericArguments());
                }
                catch (System.ArgumentException)
                {
                    //Console.WriteLine(ex);
                }
                catch (System.BadImageFormatException)
                {
                    //Console.WriteLine(ex);
                }
                catch (System.NotSupportedException)
                {
                    //Console.WriteLine(ex);
                }
                if (mb == null)
                {
                    mb = ResolveMethodTokenOnFail1(methodIndex, token);
                }
                if (mb == null)
                {
                    mb = ResolveMethodTokenOnFail2(methodIndex, token);
                }
                if (mb == null)
                {
                    result = InvalidHandle(new StringBuilder("ResolveMethodToken(").Append(methodIndex).Append(token).Append(")").ToString());
                }
                else
                {
                    ClassInfo clsInfo = AddIfNeeded(new ClassInfo(mb.DeclaringType));
                    int clsIndex = classes[clsInfo];
                    ClassGetDeclaredStaticMethods(clsIndex);
                    ClassGetDeclaredInstanceMethods(clsIndex);
                    MethodInfo m = new MethodInfo(mb, mb.DeclaringType);
                    AddIfNeeded(m);
                    result = methods[m];
                }
                methodTokensCache.Add(key, result);
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 ResolveFieldToken(int methodIndex, int token)
        {
            Int32 result;
            if (DO_PROFILING)
                StartProfiling();
            MethodInfo mi = methods[methodIndex];
            System.Reflection.FieldInfo resolvedField = null;
            try
            {
                resolvedField = mi.Method.Module.ResolveField(token, mi.Method.DeclaringType.GetGenericArguments(), mi.Method.GetGenericArguments());
            }
            catch (System.ArgumentException)
            {
                //Console.WriteLine(ex);
            }
            catch (System.BadImageFormatException)
            {
                //Console.WriteLine(ex);
            }
            catch (System.NotSupportedException)
            {
                //Console.WriteLine(ex);
            }
            if (resolvedField == null)
            {
                resolvedField = ResolveFieldTokenOnFail1(methodIndex, token);
            }
            if (resolvedField == null)
            {
                resolvedField = ResolveFieldTokenOnFail2(methodIndex, token);
            }
            if (resolvedField == null)
            {
                result = InvalidHandle(new StringBuilder("ResolveField(").Append(methodIndex).Append(",").Append(token).Append(")").ToString());
            }
            else
            {
                FieldInfo resolvedFI = new FieldInfo(resolvedField);
                AddIfNeeded(resolvedFI);
                result = fields[resolvedFI];
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean FieldIsStatic(int fieldIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = fields[fieldIndex].IsStatic;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 FieldGetType(int fieldIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            ClassInfo ci = AddIfNeeded(fields[fieldIndex].Type);
            Int32 result = classes[ci];
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 FieldDeclaringClass(int fieldIndex)
        {
            Int32 result;
            if (DO_PROFILING)
                StartProfiling();
            ClassInfo declaringClass = AddIfNeeded(fields[fieldIndex].DeclaringClass);
            result = classes[declaringClass];
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Boolean ClassIsValueType(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Boolean result = classes[clsIndex].IsValueType;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 MethodGetLocalVariableType(int methodIndex, int localIndex)
        {
            Int32 result;
            if (DO_PROFILING)
                StartProfiling();
            ClassInfo ci = methods[methodIndex].GetLocalVariableType(localIndex);
            if (ci == null)
            {
                result = InvalidHandle(new StringBuilder("MethodGetLocalVariableType(").Append(methodIndex).Append(",").Append(localIndex).Append(")").ToString());
            }
            else
            {
                ci = AddIfNeeded(ci);
                result = classes[ci];
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public String FieldGetName(int fieldIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            String result = fields[fieldIndex].Name;
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 ResolveTypeToken(int methodIndex, int token)
        {
            Int32 result;
            if (DO_PROFILING)
                StartProfiling();
            /* 
             * Ugly piece of code that compensates for our inability to resolve a token corresponding to an exception through the module corresponding
             * to the class that catches the exception.
             */
            if (exceptionTokens.ContainsKey(token))
            {
                result = exceptionTokens[token];
            }
            else
            {
                MethodInfo mi = methods[methodIndex];
                System.Type resolvedType = null;
                try
                {
                    resolvedType = mi.Method.Module.ResolveType(token, mi.Method.DeclaringType.GetGenericArguments(), mi.Method.GetGenericArguments());
                }
                catch (System.ArgumentOutOfRangeException)
                {
                    //Console.WriteLine(ex);
                }
                catch (System.ArgumentException)
                {
                    //Console.WriteLine(ex);
                }
                catch (System.BadImageFormatException)
                {
                    //Console.WriteLine(ex);
                }
                catch (System.NotSupportedException)
                {
                    //Console.WriteLine(ex);
                }
                if (resolvedType == null)
                {
                    resolvedType = ResolveTypeOnFail1(methodIndex, token);
                }
                if (resolvedType == null)
                {
                    resolvedType = ResolveTypeOnFail2(methodIndex, token);
                }
                if (resolvedType == null)
                {
                    result = InvalidHandle(new StringBuilder("ResolveType(").Append(methodIndex).Append(",").Append(token).Append(")").ToString());
                }
                else
                {
                    ClassInfo resolvedCI = AddIfNeeded(new ClassInfo(resolvedType));
                    result = classes[resolvedCI];
                }
            }
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        private System.Type ResolveTypeOnFail1(int methodIndex, int token)
        {
            MethodInfo mi = methods[methodIndex];
            System.Type result = null;
            try
            {
                result = mi.Method.Module.ResolveType(token, mi.Method.DeclaringType.GetGenericArguments(), new Type[0]);
            }
            catch (System.NotSupportedException)
            {
            }
            catch (System.BadImageFormatException)
            {
            }
            catch (ArgumentOutOfRangeException)
            {
            }
            catch (System.ArgumentException)
            {
            }
            return result;
        }

        private System.Type ResolveTypeOnFail2(int methodIndex, int token)
        {
            MethodInfo mi = methods[methodIndex];
            System.Type result = null;
            try
            {
                result = mi.Method.Module.ResolveType(token);
            }
            catch (System.NotSupportedException)
            {
            }
            catch (System.BadImageFormatException)
            {
            }
            catch (ArgumentOutOfRangeException)
            {
            }
            catch (System.ArgumentException)
            {
            }
            return result;
        }

        public Int32 WithGetBase(int index)
        {
            if (DO_PROFILING)
                StartProfiling();
            ClassInfo ci = classes[index];
            Type genericTypeDef = ci.Type.GetGenericTypeDefinition();
            ClassInfo withBaseCls = AddIfNeeded(new ClassInfo(genericTypeDef));
            Int32 result = classes[withBaseCls];
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        public Int32 PointerGetReferrent(int clsIndex)
        {
            if (DO_PROFILING)
                StartProfiling();
            Int32 result = GetElementType(clsIndex);
            if (DO_PROFILING)
                EndProfiling();
            return result;
        }

        private Int32 GetElementType(int clsIndex)
        {
            ClassInfo arrayClass = classes[clsIndex];
            ClassInfo elementCls = AddIfNeeded(new ClassInfo(arrayClass.Type.GetElementType()));
            return classes[elementCls];
        }

        private String getShortName(String fullName)
        {
            int iComma = fullName.IndexOf(",");
            if (iComma == -1)
            {
                return null;
            }
            else
            {   // Try finding it based on its Short Name only
                String shortName = fullName.Substring(0, iComma);
                return shortName;
            }
        }
	#endregion
    }
}
