using System;
using System.Collections.Generic;
using System.Text;
using System.Reflection;

namespace CSharpImpl
{
    public class MethodInfo
    {
        private readonly System.Reflection.MethodBase method;
		
		private readonly Type declaringClass;
		
		private readonly int numberOfParameters;
		
        #region CTOR
        public MethodInfo(System.Reflection.MethodBase method, Type declaringClass)
        {
            this.method = method;
			this.numberOfParameters = method.GetParameters().Length;
			this.declaringClass = declaringClass;
        }
        #endregion

        #region Methods
        public override string ToString()
        {
            return method.ToString();
        }

        public ClassInfo GetLocalVariableType(int localIndex)
        {
            int numLocals = method.GetMethodBody().LocalVariables.Count;
            return localIndex < numLocals ? new ClassInfo(method.GetMethodBody().LocalVariables[localIndex].LocalType) : null;
        }

        public override bool Equals(object obj)
        {
            if (obj is MethodInfo)
            {
                MethodInfo other = (MethodInfo) obj;
                if (other.method.Equals(method))
                {
                    return true;
                }
            }
            return false;
        }

        public override int GetHashCode()
        {
            return method.GetHashCode();
        }

        public ClassInfo GetParameterType(Int32 paramIndex)
        {
            return new ClassInfo(method.GetParameters()[paramIndex].ParameterType);
        }

        /**
        * get the GetAccessMask attribute for the method, with the following
        * possible values: 
        *   0 for CompilerControlled, 
        *   1 for Private, 
        *   2 for FamANDAssem, 
        *   3 for Assem, 
        *   4 for Family, 
        *   5 for FamORAssem, 
        *   6 for Public
        * 
        * @param method
        *            
        * @return the access mask
        */
        public Int32 GetAccessMask()
        {
            int result = 0;
            if (method.IsPrivate)
                result = 1;
            else if (method.IsFamilyAndAssembly)
                result = 2;
            else if (method.IsAssembly)
                result = 3;
            else if (method.IsFamily)
                result = 4;
            else if (method.IsFamilyOrAssembly)
                result = 5;
            else if (method.IsPublic)
                result = 6;

            return result;
        }
        #endregion

        #region Properties
        public Vector<AttributeInfo> Attributes
        {
            get
            {
				IList<System.Reflection.CustomAttributeData> attributes;
				PropertyInfo p = new ClassInfo(method.DeclaringType).getPropertyForMethod(method);
				if (p != null) {
					attributes = CustomAttributeData.GetCustomAttributes(p);
				} else {
					attributes = CustomAttributeData.GetCustomAttributes(method);
				}
				
				Vector<AttributeInfo> result = new Vector<AttributeInfo>(attributes.Count);
				
				foreach (System.Reflection.CustomAttributeData att in attributes)
                {
                    result.Add(new AttributeInfo(att));
                }
                
				return result;
            }
        }

        public Boolean IsNative
        {
            get
            {
                return ((method.GetMethodBody() == null) && (!IsAbstract));
            }
        }

        public Boolean IsStatic
        {
            get
            {
                return method.IsStatic;
            }
        }

        public Int32 MaxStackHeight
        {
            get
            {
                return method.GetMethodBody().MaxStackSize;
            }
        }

        public Boolean IsPublic
        {
            get
            {
                return method.IsPublic;
            }
        }

        public IList<ExceptionHandlingClause> ExceptionEntries
        {
            get
            {
                System.Reflection.MethodBody body = method.GetMethodBody();
                return (body == null) ? new List<ExceptionHandlingClause>(0) : body.ExceptionHandlingClauses;
            }
        }

            public Int32 HeaderSize
        {
            get
            {
                MethodBody body = method.GetMethodBody();
                if (body != null)
                {
                    const int TINY_METHOD_HEADER_SIZE = 1;
                    const int FAT_METHOD_HEADER_SIZE = 12;
                    /**
                     * Tiny method body:
                     * -    Header size is 1 bytes.
                     * -    Used when code size is less than 64 bytes, maximum stack size is less than 8, the method has no local 
                     *      variables, and there are no exceptions.
                     * Fat method:
                     * -    Header size is 12 bytes.
                     */
                    if ((body.MaxStackSize < 8) && (body.LocalVariables.Count == 0) && (body.ExceptionHandlingClauses.Count == 0))
                    {
                        return TINY_METHOD_HEADER_SIZE;
                    }
                    else
                    {
                        return FAT_METHOD_HEADER_SIZE;
                    }
                }
                else
                {
                    return -1;
                }
            }
        }

        public Int32 LocalVariablesCount
        {
            get
            {
                System.Reflection.MethodBody body = method.GetMethodBody();
                System.Diagnostics.Debug.Assert(body != null);
                return body.LocalVariables.Count;
            }
        }

        public Boolean IsPrivate
        {
            get
            {
                return method.IsPrivate;
            }
        }

        public Boolean IsInit
        {
            get
            {
                return method.IsConstructor;
            }
        }

        public Boolean IsAbstract
        {
            get
            {
                return method.IsAbstract;
            }
        }

        public Boolean IsFinal
        {
            get
            {
                return method.IsFinal;
            }
        }

        public System.Reflection.MethodBase Method
        {
            get
            {
                return method;
            }
        }

        public String Name
        {
            get
            {
                return method.Name;
            }
        }

        public ClassInfo DeclaringClass
        {
            get
            {
                return new ClassInfo(declaringClass);
            }
        }

        public Int32 NumberOfParameters
        {
            get
            {
                return numberOfParameters;
            }
        }

        public ClassInfo ReturnType
        {
            get
            {
                return (method is System.Reflection.MethodInfo) ? new ClassInfo(((System.Reflection.MethodInfo) method).ReturnType) : new ClassInfo(typeof(void));
            }
        }
		
		public Int32 NumberOfOutParameters
		{
			get {
				Int32 outParams = 0;
				System.Reflection.ParameterInfo[] pms = method.GetParameters();
				for(int i = 0; i < pms.GetLength(0); i++) {
					if (pms[i].IsOut || pms[i].IsRetval) {
						outParams++;
					}
				}
				return outParams;
			}
		}

		public Vector<AttributeInfo> GetParameterAnnotations(Int32 paramIndex)
		{
			IList<System.Reflection.CustomAttributeData> attributes=  CustomAttributeData.GetCustomAttributes(method.GetParameters()[paramIndex]);
			Vector<AttributeInfo> result = new Vector<AttributeInfo>(attributes.Count);
			foreach (System.Reflection.CustomAttributeData att in attributes)
			{
				result.Add(new AttributeInfo(att));
			}
			return result;
		}
		#endregion
    }
}
