
extern "C" {
#include "il_dumpasm.h"
#include "il_image.h"
#include "il_debug.h"
#include "il_system.h"
#include <string.h>
}

#include <jni.h>

#include <string>
#include <set>

#include "two_way_map.h"

#include "com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface.h"

#include "Exceptions.h"

static bool DEBUG;

static jmethodID boxMethod;
static jmethodID unboxMethod;

#define BOX(_I) env->CallObjectMethod(cls, boxMethod, (jint) _I)
#define UNBOX(_I) env->CallIntMethod(cls, unboxMethod, _I)

using namespace std;

static const char *get_image_type_str(ILImage *image) {
  switch (ILImageType(image)) {
  case IL_IMAGETYPE_DLL: return "DLL";
  case IL_IMAGETYPE_EXE: return "EXE";
  case IL_IMAGETYPE_JAVA: return "Java";
  case IL_IMAGETYPE_OBJ: return "OBJ";
  default: return "Unknown";
  }
}

template<class data_t>
static void for_each_class_rec(ILImage *image, ILClass *cls, void (*classf)(ILClass *, data_t), data_t data) {
  (*classf)(cls, data);
  ILNestedInfo *nested = NULL;
  while((nested = ILClassNextNested(cls, nested)) != 0) {
    for_each_class_rec(image, ILNestedInfoGetChild(nested), classf, data);
  }
}

template<class data_t>
static void for_each_class(ILImage *image, void (*classf)(ILClass *, data_t), data_t data) {
  int numTokens = ILImageNumTokens(image, IL_META_TOKEN_TYPE_DEF);
  for(int token = 1; token <= numTokens; ++token) {
    ILClass *c = 
      (ILClass*)ILImageTokenInfo(image, IL_META_TOKEN_TYPE_DEF | token);
    for_each_class_rec(image, c, classf, data);
  }
}

template<class data_t, class member_t>
static void for_each_member(int kind, ILClass *cls, void (*methodf)(member_t *, data_t), data_t data) {
  ILMember *member = 0;
  while((member = ILClassNextMember(cls, member)) != 0) {
    if (ILMemberGetKind(member) == kind) {
      (*methodf)((member_t*)member, data);
    }
  }
}
/*
static ILClass *get_super_class(ILClass *cls) {
  return ILProgramItemToClass(ILClass_Parent(cls));
}
*/

static int find_class_f(void *token, void *name) {
  ILClass *cls = (ILClass*)token;

  if (cls != NULL) {
    const char *cls_name = ((const char **) name)[0];
    const char *pkg_name = ((const char **) name)[1];
    if (strcmp(ILClass_Name(cls), (const char *)cls_name) == 0) {
      const char *cls_pkg = ILClass_Namespace(cls);

      // fprintf(stderr, "looking at %s\n", cls_pkg); fflush(stderr);

      if (pkg_name == NULL || cls_pkg == NULL) {
	if (cls_pkg == NULL && pkg_name == NULL) {
	  return 0;
	}
      } else {
	if (strcmp(pkg_name, cls_pkg) == 0) {
	  return 0;
	}
      }
    }
  }

  return 1;
}

static ILClass *
find_class(ILImage *image, const char *name, const char *pkg_name) 
{
  const char *x[] = {name, pkg_name};
  ILClass *cls = (ILClass *)
    ILImageSearchForToken(image, IL_META_TOKEN_TYPE_DEF, find_class_f, x);

  return cls;
}

static ILMethod *find_method(ILClass *cls, const char *name, ILType *sig) {
  return (ILMethod *)
    ILClassNextMemberMatch(cls, NULL, IL_META_MEMBERKIND_METHOD, name, sig);
}

static ILField *find_field(ILClass *cls, const char *name, ILType *sig) {
  return (ILField *)
    ILClassNextMemberMatch(cls, NULL, IL_META_MEMBERKIND_FIELD, name, sig);
}

static void dump_method(ILMethod *m, ILClass *cls) {
  const char *cls_name = ILClass_Name(cls);
  const char *method_name = ILMethod_Name(m);
  ILType *method_sig = ILMethod_Signature(m);

  ILMethod *x = find_method(cls, method_name, method_sig);
  const char *y = ILMethod_Name(x);
  
  fprintf(stdout, "class %s, method %s (%s)\n", cls_name, method_name, y);
  fprintf(stdout, "  %s\n", ILTypeToName(method_sig));
}

static void dump_field(ILField *m, ILClass *cls) {
  const char *cls_name = ILClass_Name(cls);
  const char *field_name = ILField_Name(m);
  ILType *field_type = ILField_Type(m);

  ILField *x = find_field(cls, field_name, field_type);
  const char *y = ILField_Name(x);

  fprintf(stdout, "class %s, field %s (%s)\n", cls_name, field_name, y);
  fprintf(stdout, "  %s\n", ILTypeToName(field_type));
}

static void dump_class(ILClass *cls, ILImage *image) {
  const char *cls_name = ILClass_Name(cls);
  const char *pkg_name = ILClass_Namespace(cls);
  ILClass *x = find_class(image, cls_name, pkg_name);

  ILDebugContext *dbg = ILDebugCreate(image);
  ILUInt32 line, col;
  const char *fname = ILDebugGetLineInfo(dbg, ILClass_Token(x), 0, &line, &col);
  ILDebugDestroy(dbg);

  fprintf(stdout, "class %s (source file: %s, line %d, col %d)\n", cls_name, fname, line, col);

  for_each_member(IL_META_MEMBERKIND_METHOD, cls, dump_method, cls);

  for_each_member(IL_META_MEMBERKIND_FIELD, cls, dump_field, cls);
}

static bool
get_method_info(ILMethod *method, ILMethodCode *code, ILException **exceptions) 
{
  if(!ILMethodGetCode(method, code)) {
    return false;
  }

  if(!ILMethodGetExceptions(method, code, exceptions)) {
    return false;
  }

  return true;
}

static const int flags = IL_LOADFLAG_IGNORE_ERRORS;

static ILContext *context;

static int nextHandle = 1;

static two_way_map<int,ILImage*> images;

static two_way_map<int,ILClass*> classes;

static two_way_map<int,ILAttribute*> attributes;

static two_way_map<string,int> built_in_types;

static two_way_map<int,ILType*> allTypes;

static two_way_map<int,ILMethod*> methods;

static two_way_map<int,ILField*> fields;

static two_way_map<string,ILImage*> images_by_name;

static two_way_map<string,ILImage*> images_by_file_name;

static void built_in_type(const char *name, ILType *type) {
  built_in_types[name] = nextHandle;
  allTypes[nextHandle++] = type;
}

static int typeId(ILClass *cls) {
  ILClass *x = ILClassResolve(cls);
  if (x != NULL) {
    cls = x;
  }

  if (classes.contains(cls)) {
    return classes[cls];
  } else {
    classes[nextHandle] = cls;
    allTypes[nextHandle] = ILType_FromClass( cls );

    return nextHandle++;
  }
}

static jobject boxType(JNIEnv *env, jobject cls, ILType *type) {
  if (ILType_IsClass(type)) {
    ILClass *x = ILType_ToClass(type);
    x = ILClassResolve(x);
    if (x != NULL) {
      type = ILClassToType(x);
    }
  } else if (ILType_IsValueType(type)) {
    ILClass *x = ILType_ToValueType(type);
    x = ILClassResolve(x);
    if (x != NULL) {
      type = ILClassToType(x);    
    }
  }

  if (allTypes.contains(type)) {
    int handle = allTypes[type]; 
    return BOX(handle);
  } else {
    if (ILType_IsClass(type)) {
      classes[nextHandle] = ILType_ToClass(type);
    } else if (ILType_IsValueType(type)) {
      classes[nextHandle] = ILType_ToValueType(type);
    }
    allTypes[nextHandle] = type;

    return BOX(nextHandle++);
  }
}

JNIEXPORT void JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_init(
	      JNIEnv *env, 
	      jclass cls) 
{
  TRY(exp, env)

  context = ILContextCreate();
  if(!context) {
    THROW(exp, "out of memory in native code");
  }

  jclass constants = env->FindClass("com/ibm/wala/dotnet/Constants"); 
  THROW_ANY_EXCEPTION(exp);
  jfieldID traceFlag = env->GetStaticFieldID(constants, "TRACE_LOW_LEVEL", "Z");
  THROW_ANY_EXCEPTION(exp);
  DEBUG = env->GetStaticBooleanField(constants, traceFlag);
  THROW_ANY_EXCEPTION(exp);

  boxMethod = env->GetMethodID(cls, "box", "(I)Ljava/lang/Integer;");
  THROW_ANY_EXCEPTION(exp);
  unboxMethod = env->GetMethodID(cls, "unbox", "(Ljava/lang/Integer;)I");
  THROW_ANY_EXCEPTION(exp);

  built_in_type("System/Void",    ILType_Void);
  built_in_type("System/Boolean", ILType_Boolean);     
  built_in_type("System/Char",    ILType_Char); 
  built_in_type("System/Byte",    ILType_Int8);    
  built_in_type("System/SByte",   ILType_UInt8);      
  built_in_type("System/Int16",   ILType_Int16);    
  built_in_type("System/UInt16",  ILType_UInt16);     
  built_in_type("System/Int32",   ILType_Int32);     
  built_in_type("System/UInt32",  ILType_UInt32);     
  built_in_type("System/Int64",   ILType_Int64);     
  built_in_type("System/UInt64",  ILType_UInt64);     
  built_in_type("System/Single", ILType_Float32);     
  built_in_type("System/Double", ILType_Float64);     
  built_in_type("System/String",  ILType_String);     
  built_in_type("System/Int",     ILType_Int);      
  built_in_type("System/UInt",    ILType_UInt);    
  built_in_type("System/Float",   ILType_Float);     

  built_in_type("System/TypedReference", ILType_TypedRef);     

  CATCH()
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_findImage(
               JNIEnv *env, 
	       jobject cls, 
	       jstring str,
	       jboolean isSystemImage) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  TRY(exp, env)

  const char *file_name = env->GetStringUTFChars(str,NULL);
  THROW_ANY_EXCEPTION(exp);

  if (images_by_file_name.contains(file_name)) {
    return BOX(images[ images_by_file_name[ file_name ] ]);
  }

  FILE *stream = fopen(file_name, "rb");
  if (! stream) {
    char err[255];
    sprintf(err, "cannot open %s\n", file_name);
    THROW(exp, err);
  }

  ILImage *image = NULL;
  int loadError = ILImageLoad(stream, file_name, context, &image, flags);
  if(loadError != 0) {
    fclose(stream);
    char err[255];
    sprintf(err, "%s: %s\n", file_name, ILImageLoadError(loadError));
    THROW(exp, err);
  }

  fclose(stream);

  // fprintf(stderr, "image for %s is %s\n", file_name, ILImageGetAssemblyName(image));
  if (images.contains(image)) {
    fprintf(stderr, "already have image for %s\n", file_name);
  }

  images[nextHandle] = image;
  images_by_name[ILImageGetAssemblyName(image)] = image;
  images_by_file_name[file_name] = image;

  if (isSystemImage) {
    jclass constants = env->FindClass("com/ibm/wala/dotnet/Constants");
    THROW_ANY_EXCEPTION(exp);

    jfieldID systemNameField = env->GetStaticFieldID(constants, "systemNameStr", "Ljava/lang/String;");
    THROW_ANY_EXCEPTION(exp);

    jstring systemNameStr = (jstring) env->GetStaticObjectField(constants, systemNameField);
    THROW_ANY_EXCEPTION(exp);

    const char *system_name = env->GetStringUTFChars(systemNameStr,NULL);
    THROW_ANY_EXCEPTION(exp);

    images_by_name[system_name] = image;

    images_by_name["Primordial"] = image;

    images_by_name["$Synthetic"] = image;

    ILContextSetSystem(context, image);
  }

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  return BOX(nextHandle++);

  CATCH()

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  return NULL;
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_getImage
  (JNIEnv *env, 
   jobject cls, 
   jstring imageName)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  TRY(exp, env)

  const char *image_name = env->GetStringUTFChars(imageName,NULL);
  THROW_ANY_EXCEPTION(exp);
  
  if (! images_by_name.contains(image_name)) {
    char buf[255];
    sprintf(buf, "cannot find image %s", image_name);
    THROW(exp, buf);
  }

  ILImage *image = images_by_name[image_name];

  if (! images.contains(image)) {
    char buf[255];
    sprintf(buf, "cannot find hande for image %s", image_name);
    THROW(exp, buf);
  }

  int imageHandle = images[image];

  return BOX(imageHandle);

  CATCH()

  return NULL;
}

JNIEXPORT jstring JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_imageGetAssemblyName(
                          JNIEnv *env, 
			  jobject cls,
			  jobject imageHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  TRY(exp, env)

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  if (! images.contains(UNBOX(imageHandle))) {
    char buf[255];
    sprintf(buf, "cannot find image for handle %d", UNBOX(imageHandle));
    THROW(exp, buf);
  }

  ILImage *image = images[UNBOX(imageHandle)];

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  const char *nm = ILImageGetAssemblyName(image);

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return env->NewStringUTF(nm); 

  CATCH()

  return NULL; 
}

static jstring get_file_name(JNIEnv *env, ILProgramItem *item, int bytecodeOffset) {
  ILImage *image = ILProgramItemGetImage(item);

  ILDebugContext *dbg = ILDebugCreate(image);
  ILUInt32 line, col;
  const char *nm = ILDebugGetLineInfo(dbg, ILClass_Token(item), bytecodeOffset, &line, &col);

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  jstring result = env->NewStringUTF(nm);  

  ILDebugDestroy(dbg);

  return result;
}

JNIEXPORT jstring JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetSourceFileName
  (JNIEnv *env,
   jobject cls,
   jobject methodHandle,
   jint bytecodeOffset) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  return get_file_name(env, ILToProgramItem((ILMethod*)methods[UNBOX(methodHandle)]), bytecodeOffset);
}

JNIEXPORT jstring JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classGetSourceFile(
                       JNIEnv *env, 
		       jobject cls, 
		       jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  return get_file_name(env, ILToProgramItem((ILClass*)classes[UNBOX(classHandle)]), 0);
}

JNIEXPORT jobjectArray JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classGetDeclaredInterfaces(
                                       JNIEnv *env, 
				       jobject cls, 
				       jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILImplements *implClause = NULL;
  ILClass *c = classes[UNBOX(classHandle)];
  set<ILClass*> ifaces;

  while ((implClause = ILClassNextImplements(c, implClause)) != NULL) {
    ILClass *iface = ILProgramItemToClass(ILImplementsGetInterface(implClause));
    if (ILClass_SynType(iface) == NULL) {
      ILClass *real = ILClassResolve(iface);
      if (real != NULL) {
	iface = real;
      }
      ifaces.insert(iface);
    }
  }

  int i = 0;
  jclass javaLangIntegerCls = env->FindClass("java/lang/Integer"); 
  jobjectArray arr = 
    env->NewObjectArray(ifaces.size(), javaLangIntegerCls, NULL);
  for (set<ILClass*>::iterator it=ifaces.begin(); it != ifaces.end(); it++) {
    env->SetObjectArrayElement(arr, i++, BOX(typeId(*it)));
  }

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return arr;
}

typedef struct { bool wantStatic; set<int> result; } _qd;

static void findDesiredField(ILField *member, _qd* data) {
  if (ILField_IsStatic(member) == data->wantStatic) {
    if (fields.contains(member)) {
      data->result.insert(fields[member]);
    } else {
      fields[nextHandle] = member;
      data->result.insert(nextHandle++);
    }
  }
}

static void findDesiredMethod(ILMethod *member, _qd* data) {
  if ((ILMethod_CallConv(member)&IL_META_CALLCONV_HASTHIS) ^ data->wantStatic) {
    if (methods.contains(member)) {
      data->result.insert(methods[member]);
    } else {
      methods[nextHandle] = member;
      data->result.insert(nextHandle++);
    }
  }
}

static void findStaticCtor(ILMethod *member, ILMethod **result) {
  if (ILMethodIsStaticConstructor(member)) {
    *result = member;
  }
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classGetStaticCtor(
                               JNIEnv *env, 
			       jobject cls, 
			       jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  TRY(exp, env)

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *result = NULL;
  for_each_member(IL_META_MEMBERKIND_METHOD, classes[UNBOX(classHandle)], findStaticCtor, &result);

  if (result == NULL) {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return BOX(-1);
  } else if (methods.contains(result)) {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return BOX(methods[result]);
  } else {
    methods[nextHandle] = result;
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return BOX(nextHandle++);
  }

  CATCH()

  return NULL;
}

template<class member_t>
static jobjectArray 
findDesiredMembers(JNIEnv *env, 
		   jobject cls,
		   ILClass *c, 
		   int kind, 
		   bool isStatic, 
		   void (*methodf)(member_t *, _qd*)) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  _qd data;
  data.wantStatic = isStatic;

  for_each_member(kind, c, methodf, &data);

  int i = 0;
  jclass javaLangIntegerCls = env->FindClass("java/lang/Integer"); 
  jobjectArray arr = 
    env->NewObjectArray(data.result.size(), javaLangIntegerCls, NULL);
  jint buf[ data.result.size() ];
  for (set<int>::iterator it=data.result.begin(); 
       it != data.result.end(); 
       it++) 
  {
    env->SetObjectArrayElement(arr, i++, BOX(*it));
  }

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return arr;
}

JNIEXPORT jobjectArray JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classGetDeclaredStaticFields(
                                         JNIEnv *env, 
					 jobject cls, 
					 jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return findDesiredMembers(env, cls, classes[UNBOX(classHandle)], IL_META_MEMBERKIND_FIELD, true, findDesiredField);
}

JNIEXPORT jobjectArray JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classGetDeclaredInstanceFields(
                                           JNIEnv *env, 
					   jobject cls, 
					   jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return findDesiredMembers(env, cls, classes[UNBOX(classHandle)], IL_META_MEMBERKIND_FIELD, false, findDesiredField);
}

JNIEXPORT jobjectArray JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classGetDeclaredStaticMethods(
                                         JNIEnv *env, 
					 jobject cls, 
					 jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return findDesiredMembers(env, cls, classes[UNBOX(classHandle)], IL_META_MEMBERKIND_METHOD, true, findDesiredMethod);
}

JNIEXPORT jobjectArray JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classGetDeclaredInstanceMethods(JNIEnv *env, 
					   jobject cls, 
					   jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return findDesiredMembers(env, cls, classes[UNBOX(classHandle)], IL_META_MEMBERKIND_METHOD, false, findDesiredMethod);
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_findClass(
                      JNIEnv *env, 
		      jobject cls, 
		      jobject imageHandle, 
		      jstring className, 
		      jstring packageName) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  const char *cls_name = env->GetStringUTFChars(className,NULL);

  if (DEBUG) { fprintf(stdout, "at %s, %d: %s\n", __FILE__, __LINE__, cls_name); fflush(stdout); }

  char *pkg_name = NULL;
  if (packageName != NULL) {
    pkg_name = strdup(env->GetStringUTFChars(packageName,NULL));
    char *i;
    while ((i = strchr(pkg_name, '/')) != NULL) {
      *i = '.';
    }
  }

  char pnm[ strlen(cls_name) + 8 ];
  sprintf(pnm, "System/%s", cls_name);
  if (pkg_name != NULL && strcmp("System", pkg_name) == 0 && built_in_types.contains(pnm)) {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return BOX(built_in_types[pnm]);
  
  } else {
    if (DEBUG) { fprintf(stdout, "at %s, %d: %s, %s\n", __FILE__, __LINE__, pkg_name, cls_name); fflush(stdout); }

    if (! images.contains(UNBOX(imageHandle))) {
      return BOX(-1);
    }

    ILClass *x = find_class(images[UNBOX(imageHandle)], cls_name, pkg_name);
    if (x == NULL) {
      x = ILClassResolveSystem(images[UNBOX(imageHandle)], NULL, cls_name, pkg_name);
    }

    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    if (x != NULL) {
      if (classes.contains(x)) {
	if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
	return BOX(classes[x]);
      } else {
	classes[nextHandle] = x;
	allTypes[nextHandle] = ILType_FromClass( x );
	if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
	return BOX(nextHandle++);
      }
    } else {
      if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
      return BOX(-1);
    }
  }
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_findField(
                      JNIEnv *env, 
		      jobject cls, 
		      jobject classHandle,
		      jstring fieldName,
		      jobject fieldTypeHandle) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  const char *method_name = env->GetStringUTFChars(fieldName,NULL);

  ILField *f = find_field(classes[UNBOX(classHandle)], method_name, allTypes[ UNBOX(fieldTypeHandle) ]);
  if (f != NULL) {
    if (fields.contains(f)) {
      if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
      return BOX(fields[f]);
    } else {
      fields[nextHandle] = f;
      if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
      return BOX(nextHandle++);
    }
  } else {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return BOX(-1);
  }

}

static jobject
findMethodInternal(JNIEnv *env, 
		   jobject cls, 
		   jobject classHandle,
		   jstring methodName,
		   jobjectArray sigHandles,
		   ILUInt32 callConv) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  const char *method_name = env->GetStringUTFChars(methodName,NULL);

  jsize num_params = env->GetArrayLength(sigHandles);

  int thisType = UNBOX(env->GetObjectArrayElement(sigHandles, 0));
  ILType *method = ILTypeCreateMethod(context, allTypes[thisType]);

  ILTypeSetCallConv(method, callConv);

  for(int i = 1; i < num_params; i++) {
    ILTypeAddParam(context, 
		   method,
		   allTypes[UNBOX(env->GetObjectArrayElement(sigHandles, i))]);
  }

  ILMethod *m = find_method(classes[UNBOX(classHandle)], method_name, method);
  if (m != NULL) {
    if (methods.contains(m)) {
      if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
      return BOX(methods[m]);
    } else {
      methods[nextHandle] = m;
      if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
      return BOX(nextHandle++);
    }
  } else {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return BOX(-1);
  }
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_findInstanceMethod(
                               JNIEnv *env, 
			       jobject cls, 
			       jobject classHandle,
			       jstring methodName,
			       jobjectArray sigHandles) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return findMethodInternal(env, 
			    cls, 
			    classHandle,
			    methodName,
			    sigHandles,
			    IL_META_CALLCONV_HASTHIS);
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_findClassMethod(
                               JNIEnv *env, 
			       jobject cls, 
			       jobject classHandle,
			       jstring methodName,
			       jobjectArray sigHandles) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return findMethodInternal(env, 
			    cls, 
			    classHandle,
			    methodName,
			    sigHandles,
			    IL_META_CALLCONV_DEFAULT);
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classIsPublic(
                       JNIEnv *env,
		       jobject cls,
		       jobject classHandle) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ILClass_IsPublic( classes[UNBOX(classHandle)] );
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classIsPrivate(
                       JNIEnv *env,
		       jobject cls,
		       jobject classHandle) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ILClass_IsPrivate( classes[UNBOX(classHandle)] );
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classIsAbstract(
                      JNIEnv *env,
		      jobject cls,
		      jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ILClass_IsAbstract( classes[UNBOX(classHandle)] );
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classIsInterface(
                  JNIEnv *env,
		  jobject cls,
		  jobject classHandle) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ILClass_IsInterface( classes[UNBOX(classHandle)] );
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classIsValueType(
                  JNIEnv *env,
		  jobject cls,
		  jobject typeHandle) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  TRY(exp,env)

  bool result;

  if (classes.contains(UNBOX(typeHandle))) {
    if (DEBUG) { fprintf(stdout, "class %d at %s, %d\n", UNBOX(typeHandle), __FILE__, __LINE__); fflush(stdout); }
    return ILClassIsValueType(classes[ UNBOX(typeHandle) ]);
  }

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  if (! allTypes.contains(UNBOX(typeHandle))) {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    THROW(exp, "cannot find type");
  }

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  ILType *type = allTypes[ UNBOX(typeHandle) ];

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  result = ILType_IsValueType( type );

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  return result;

  CATCH()

  return false;
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classGetParent(
                 JNIEnv *env, 
		 jobject cls, 
		 jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  int classId = UNBOX(classHandle);
  if (classes.contains(classId)) {
    ILClass *current = classes[UNBOX(classHandle)];
    ILClass *superclass = ILClass_ParentClass(current);

    if (superclass == NULL) {
      return BOX(-1);

    } else {
      if (DEBUG) {
	fprintf(stdout, "parent of %s is %s\n", ILClass_Name(current), ILClass_Name(superclass));
	fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); 
	fflush(stdout); 
      }
      
      ILClass *real = ILClassResolve(superclass);
      if (real != NULL) {
	superclass = real;
      }

      int type_id = typeId(superclass);

      return BOX(type_id==classId? -1: type_id);
    }
  } else {
    return BOX(-1);
  }
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_fieldIsFinal
  (JNIEnv *env,
   jobject cls,
   jobject fieldHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  ILField *field = fields[ UNBOX(fieldHandle) ];
  return ILField_IsInitOnly(field);
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_fieldIsStatic
  (JNIEnv *env,
   jobject cls,
   jobject fieldHandle)
{
  TRY(exp,env)

  if (DEBUG) { fprintf(stdout, "field %d at %s, %d\n", UNBOX(fieldHandle), __FILE__, __LINE__); fflush(stdout); }

  if (UNBOX(fieldHandle) == -1) {
    THROW(exp, "invalid field handle");
  }

  ILField *field = fields[ UNBOX(fieldHandle) ];
  return ILField_IsStatic(field);

  CATCH()
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_fieldIsPublic
  (JNIEnv *env,
   jobject cls,
   jobject fieldHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  ILField *field = fields[ UNBOX(fieldHandle) ];
  return ILField_IsPublic(field);
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_fieldIsProtected
  (JNIEnv *env,
   jobject cls,
   jobject fieldHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  ILField *field = fields[ UNBOX(fieldHandle) ];
  return ! ILField_IsPublic(field) && ! ILField_IsPrivate(field);
}

template<class T>
static void count_it(T *cls, int *count) {
  (*count)++;
}

static void count_methods(ILClass *cls, int *count) {
  for_each_member(IL_META_MEMBERKIND_METHOD, cls, count_it<ILMethod>, count);
}

JNIEXPORT jint JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_numberOfClasses
  (JNIEnv *env, 
   jobject cls,
   jobject imageHandle) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILImage *image = images[ UNBOX(imageHandle) ];
  int class_count = 0;
  for_each_class(image, count_it, &class_count);
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return class_count;
}

JNIEXPORT jint JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_numberOfMethods
  (JNIEnv *env, 
   jobject cls,
   jobject imageHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILImage *image = images[ UNBOX(imageHandle) ];
  int method_count = 0;
  for_each_class(image, count_methods, &method_count);
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return method_count;
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetDeclaringClass
  (JNIEnv *env,
   jobject cls, 
   jobject methodHandle)
{
  TRY(exp,env);

  if (DEBUG) { fprintf(stdout, "method %d at %s, %d\n", UNBOX(methodHandle), __FILE__, __LINE__); fflush(stdout); }

  if (UNBOX(methodHandle) == -1) {
    THROW(exp, "invalid method handle");
  }

  ILMethod *method = methods[ UNBOX(methodHandle) ];

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  ILClass *owner = ILMethod_Owner( method );

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  if (classes.contains(owner)) {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return BOX(classes[owner]);
  } else {
    classes[nextHandle] = owner;
    allTypes[nextHandle] = ILType_FromClass(owner);
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return BOX(nextHandle++);
  }

  CATCH()
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodIsAbstract
  (JNIEnv *env,
   jobject cls,
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ILMethod_IsAbstract(method);
}

JNIEXPORT jint JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetAccessMask
  (JNIEnv *env,
   jobject cls,
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return (ILMemberGetAttrs((ILMember*)method) & IL_META_METHODDEF_MEMBER_ACCESS_MASK);
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodIsClinit
  (JNIEnv *env,
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ILMethod_IsStaticConstructor(method);
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodIsFinal
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ILMethod_IsFinal(method);
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodIsInit
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ILMethod_IsConstructor(method);
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodIsNative
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ILMethod_IsNative(method);
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodIsPrivate
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ILMethod_IsPrivate(method);
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodIsPublic
  (JNIEnv *env, 
   jobject cls,
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ILMethod_IsPublic(method);
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodIsStatic
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  if (DEBUG) {
    fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout);
    const char *method_name = ILMethod_Name(method);
    ILType *method_sig = ILMethod_Signature(method);
    fprintf(stdout, "method %s %s\n", method_name, ILTypeToName(method_sig));
  }

  return ILMethod_IsStatic(method);
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodIsSynchronized
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ILMethod_IsSynchronized(method);
}

JNIEXPORT jint JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetMaxStackHeight
  (JNIEnv * env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  ILMethodCode method_code;

  int status = ILMethodGetCode(method, &method_code);
  if (status == 0) {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return -1;
  }

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return method_code.maxStack;
}

static ILType *getLocalVars(ILMethod *method) {
  ILMethodCode method_code;

  int status = ILMethodGetCode(method, &method_code);
  if (status == 0) {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return NULL;
  }

  if (method_code.localVarSig == NULL) {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return NULL;
  }

  if (!ILStandAloneSig_IsLocals(method_code.localVarSig)) {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return NULL;
  }

  ILType *locals = ILStandAloneSig_Type(method_code.localVarSig);
  if (locals == NULL) {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return NULL;
  }

  return locals;
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetLocalVariableType
  (JNIEnv *env,
   jobject cls, 
   jobject methodHandle, 
   jint localNum)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];

  ILType *locals = getLocalVars(method);

  if (locals == NULL) {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return BOX(0);
  }

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  return boxType(env, cls, ILTypeGetLocal(locals, localNum));
}

JNIEXPORT jint JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetMaxLocals
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];

  ILType *locals = getLocalVars(method);

  if (locals == NULL) {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return -1;
  }

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  return ILTypeNumLocals(locals);
}

JNIEXPORT jint JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetNumberOfParameters
  (JNIEnv *env,
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  ILType *type = ILMethod_Signature(method);
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ILTypeNumParams(type);
}

JNIEXPORT jint JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetNumberOfOutParameters
  (JNIEnv *env,
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  int out = 0;
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  ILParameter *param = NULL;
  while ((param = ILMethodNextParam(method, param)) != NULL) {
    if (ILParameter_IsOut(param) || ILParameter_IsRetVal(param)) {
      out++;
    }
  }
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return out;
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetParameterType
  (JNIEnv *env,
   jobject cls,
   jobject methodHandle,
   jint paramIndex)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  TRY(exp, env)

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  if (! methods.contains(UNBOX(methodHandle))) {
    THROW(exp, "cannot find method");
  }

  ILMethod *method = methods[ UNBOX(methodHandle) ];

  ILType *type = ILMethod_Signature(method);

  if (! type) {
    THROW(exp, "no signature for method");
  }

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return boxType( env, cls, ILTypeGetParam(type, paramIndex+1) );

  CATCH()

  return NULL;
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetReturnType
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

 ILMethod *method = methods[ UNBOX(methodHandle) ];

 ILType *type = ILMethod_Signature(method);

 ILType *returnType = ILTypeGetReturn(type);

 
 if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

 return boxType( env, cls, returnType );
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodHasExceptionHandler
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  ILMethodCode method_code;

  int status = ILMethodGetCode(method, &method_code);
  if (status == 0) {
    if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
    return false;
  }

  ILException *except;
  ILMethodGetExceptions(method, &method_code, &except);

  ILMethodFreeExceptions(except);

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return except != NULL;
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodHasLocalVariableTable
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[UNBOX(methodHandle)];
  ILImage *image = ILProgramItemGetImage(ILToProgramItem(method));

  ILDebugContext *dbg = ILDebugCreate(image);
  ILDebugIter iter;
  ILDebugIterInit(&iter, dbg, ILMethod_Token(method));
  do {
    if (iter.type==IL_DEBUGTYPE_VARS || iter.type==IL_DEBUGTYPE_VARS_OFFSETS) {
      ILDebugDestroy(dbg);
      if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
      return true;
    }
  } while (ILDebugIterNext(&iter));

  ILDebugDestroy(dbg);
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return false;
}

JNIEXPORT jint JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetLineNumber
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle, 
   jint bytecodeOffset)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[UNBOX(methodHandle)];
  ILImage *image = ILProgramItemGetImage(ILToProgramItem(method));

  ILDebugContext *dbg = ILDebugCreate(image);

  ILUInt32 line, col;

  const char *file_name = ILDebugGetLineInfo(dbg, ILMethod_Token(method), (ILUInt32) bytecodeOffset, &line, &col);

  ILDebugDestroy(dbg);
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  return line;
}

JNIEXPORT jstring JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_typeGetName(
             JNIEnv *env, 
	     jobject cls, 
	     jobject classHandle) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  TRY(exp, env)

  int h = UNBOX(classHandle);

  if (!allTypes.contains(h)) {
    char buf[255];
    sprintf(buf, "no type for handle %d", h);
    THROW(exp, buf);
  }

  char *nm = ILTypeToName(allTypes[h]);

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  jstring str = env->NewStringUTF(nm);
  ILFree(nm);

  return str;

  CATCH()

  return NULL;
}

JNIEXPORT jstring JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classGetName
  (JNIEnv *env, 
   jobject cls, 
   jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  TRY(exp, env)

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  
int h = UNBOX(classHandle);

  if (! classes.contains(h)) {
    if (allTypes.contains(h)) {
      if (built_in_types.contains(h)) {
	  return Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_primitiveGetName(env, cls, classHandle);
      } else { 
      ILType *type = allTypes[h];
      char *nm = ILTypeToName(type);
	jstring name = env->NewStringUTF(nm);
      ILFree(nm);
	return name;
      }
    } else {
      char buf[255];
      sprintf(buf, "no class for handle %d", h);
      THROW(exp, buf);
    }
  
  } else {
    ILClass *c = classes[ h ];
    const char *name = ILClass_Name(c);
    const char *ns = ILClass_Namespace(c);
    if (ns == NULL) {
      return env->NewStringUTF(name);

    } else {    
      char buf[ strlen(name) + strlen(ns) + 15 ];
      sprintf(buf, "%s/%s", ns, name);
      
      char *i;
      while ((i = strchr(buf, '.')) != NULL) {
	*i = '/';
      }

      if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

      return env->NewStringUTF(buf);    
    }
  }

  CATCH()

  return NULL;
}

JNIEXPORT jstring JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetName
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  const char *name = ILMethod_Name(method);
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return env->NewStringUTF(name);    
}

static void push_class(ILClass *cls, set<ILClass *> *list) {
  list->insert(cls);
}

JNIEXPORT jobjectArray JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_imageAllClasses
  (JNIEnv *env, 
   jobject cls, 
   jobject imageHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILImage *image = images[ UNBOX(imageHandle) ];

  set<ILClass *> classes;
  for_each_class(image, push_class, &classes);

  int i = 0;
  jclass javaLangIntegerCls = env->FindClass("java/lang/Integer"); 
  jobjectArray handles =  
    env->NewObjectArray(classes.size(), javaLangIntegerCls, NULL);
  for (set<ILClass*>::iterator it=classes.begin(); it != classes.end(); it++) {
    env->SetObjectArrayElement(handles, i++, BOX(typeId(*it)));
  }

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return handles;
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classIsPrimitive
  (JNIEnv *env, 
   jobject cls, 
   jobject classHandle) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return built_in_types.contains( UNBOX(classHandle) );
}

JNIEXPORT jstring JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_primitiveGetName
  (JNIEnv *env, 
   jobject cls,
   jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return env->NewStringUTF(((string)built_in_types[UNBOX(classHandle)]).c_str());
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classIsArray
  (JNIEnv *env,
   jobject cls, 
   jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILType *type = allTypes[ UNBOX(classHandle) ];
  return ILType_IsArray(type);
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classIsTypeVar
  (JNIEnv *env,
   jobject cls, 
   jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILType *type = allTypes[ UNBOX(classHandle) ];
  return ILType_IsTypeParameter(type);
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_arrayGetElement
  (JNIEnv *env,
   jobject cls, 
   jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILType *type = allTypes[ UNBOX(classHandle) ];
  ILType *elt = ILType_ElemType(type);
  return boxType(env, cls, elt);
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classIsReference
  (JNIEnv *env, 
   jobject cls, 
   jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILType *type = allTypes[ UNBOX(classHandle) ];

  bool result = ILType_IsRef(type);

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return result;
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_refGetReferrent
  (JNIEnv *env, 
   jobject cls, 
   jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  ILType *type = allTypes[ UNBOX(classHandle) ];
  ILType *elt = ILType_Ref(type);
  jobject ret = boxType(env, cls, elt);
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return ret;
}

JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classIsPointer
  (JNIEnv *env, 
   jobject cls, 
   jobject typeHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  ILType *type = allTypes[ UNBOX(typeHandle) ];

  bool result = ILType_IsPointer(type);

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  return result;
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_pointerGetReferrent
  (JNIEnv *env, 
   jobject cls, 
   jobject typeHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_refGetReferrent(env, cls, typeHandle);
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classGetImage
  (JNIEnv *env, 
   jobject cls, 
   jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  TRY(exp, env)

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  int h = UNBOX(classHandle);

  if (! classes.contains(h)) {
    char buf[255];
    sprintf(buf, "cannot find class for handle %d", h);
    THROW(exp, buf);
  }

  ILClass *c = classes[h];
  ILImage *image = ILProgramItemGetImage(ILToProgramItem(c));

  if (DEBUG) { 
    const char *nm = ILImageGetAssemblyName(image);
    fprintf(stdout, "image for %s is %s\n", ILClass_Name(c), nm);
    fflush(stdout); 
  }

  if (! images.contains(image)) {
    images[nextHandle++] = image;
  }
 
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return BOX(images[image]);

  CATCH()

  return BOX(-1);
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_withGetBase
  (JNIEnv *env, 
   jobject cls, 
   jobject typeHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILType *type = allTypes[ UNBOX(typeHandle) ];

  ILType *base = ILTypeGetWithMain(type);

  jobject result = boxType(env, cls, base);

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return result;
} 


JNIEXPORT jint JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_withGetNumParams
  (JNIEnv *env, 
   jobject cls, 
   jobject typeHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILType *type = allTypes[ UNBOX(typeHandle) ];

  int result = ILTypeNumWithParams(type);

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return result;
} 


JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_withGetParam
  (JNIEnv *env, 
   jobject cls, 
   jobject typeHandle,
   jint index)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILType *type = allTypes[ UNBOX(typeHandle) ];

  ILType *base = ILTypeGetWithParam(type, index+1);

  jobject result = boxType(env, cls, base);

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return result;
} 


JNIEXPORT jboolean JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classIsWith
  (JNIEnv *env,
   jobject cls,
   jobject typeHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILType *type = allTypes[ UNBOX(typeHandle) ];

  bool result = ILType_IsWith(type);

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  return result;
} 

JNIEXPORT jbyteArray JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetBytecode
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  ILMethodCode code;

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  if(!ILMethodGetCode(method, &code)) {
    return NULL;
  }

  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  jbyteArray bytes = env->NewByteArray(code.codeLen);
  env->SetByteArrayRegion(bytes, 0, code.codeLen, (const jbyte*)code.code);
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  return bytes;
}

JNIEXPORT jint JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetHeaderSize
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  TRY(exp, env)

  ILMethod *method = methods[ UNBOX(methodHandle) ];
  ILMethodCode code;

  if(!ILMethodGetCode(method, &code)) {
    THROW(exp, "no code for method");
  }

  return code.headerSize;

  CATCH()

  return -1;
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_resolveTypeToken
  (JNIEnv *env, 
   jobject cls, 
   jobject methodHandle, 
   jint tok)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  ILUInt32 token = *((ILUInt32*)&tok);

  ILProgramItem *item =
    ((ILProgramItem *)ILImageTokenInfo(ILProgramItem_Image(method), token));

  ILClass *classInfo = ILProgramItemToClass(item);

  if(classInfo) {
    classInfo = ILClassResolveToInstance(classInfo, method);

    ILClass *x = ILClassResolve(classInfo);
    if (x != NULL) {
      classInfo = x;
    }
  }

  if (classInfo) {
    return BOX(typeId(classInfo));
  } 

  ILType *type = ILProgramItemToType(item);

  if (type) {
    return boxType(env, cls, type);
  }

  return BOX(-1);
}

JNIEXPORT jobject JNICALL
 Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_resolveMethodToken
  (JNIEnv *env,
   jobject cls,
   jobject methodHandle,
   jint tok)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  ILUInt32 token = *((ILUInt32*)&tok);

  ILMethod *methodInfo;

  if((token & IL_META_TOKEN_MASK) == IL_META_TOKEN_METHOD_SPEC) {
    ILMethodSpec *mspec = 
      ILMethodSpec_FromToken(ILProgramItem_Image(method), token);

    if(!mspec) {
      return BOX(-1);
    }
	
    methodInfo = ILMethodSpecToMethod(mspec, method);

  } else {
    /* Get the token and resolve it */
    methodInfo =
      ILProgramItemToMethod((ILProgramItem *)
        ILImageTokenInfo(ILProgramItem_Image(method), token));
  }

  if(!methodInfo) {
    return BOX(-1);
  }

  ILMember *x = ILMemberResolve((ILMember*)methodInfo);
  if (x != NULL) {
    methodInfo = (ILMethod*)x;
  }

  methodInfo = (ILMethod *)
    ILMemberResolveToInstance((ILMember *)methodInfo, method);

  if (!methodInfo) {
    return BOX(-1);
  }

  if (methods.contains(methodInfo)) {
    return BOX(methods[methodInfo]);
  } else {
    methods[nextHandle] = methodInfo;
    return BOX(nextHandle++);
  }
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_resolveFieldToken
  (JNIEnv *env,
   jobject cls, 
   jobject methodHandle, 
   jint itok)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILUInt32 token = *((ILUInt32 *) &itok);
  ILMethod *method = methods[ UNBOX(methodHandle) ];
  ILField *fieldInfo;

  fieldInfo =
    ILProgramItemToField((ILProgramItem *)
      ILImageTokenInfo(ILProgramItem_Image(method), token));
  
  if(!fieldInfo) {
    return BOX(-1);
  }

  fieldInfo = (ILField *)
    ILMemberResolveToInstance((ILMember *)fieldInfo, method);

  if(!fieldInfo) {
    return BOX(-1);
  }

  if (fields.contains(fieldInfo)) {
    return BOX(fields[fieldInfo]);
  } else {
    fields[nextHandle] = fieldInfo;
    return BOX(nextHandle++);
  }
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_fieldDeclaringClass
  (JNIEnv *env,
   jobject cls, 
   jobject fieldHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILField *f = fields[ UNBOX(fieldHandle) ];
  ILClass *type = ILField_Owner(f);
  return BOX(typeId(type));
}

JNIEXPORT jstring JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_fieldGetName
  (JNIEnv *env,
   jobject cls,
   jobject fieldHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILField *f = fields[ UNBOX(fieldHandle) ];
  const char *name = ILField_Name(f);
  return env->NewStringUTF(name);
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_fieldGetType
  (JNIEnv *env,
   jobject cls,
   jobject fieldHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILField *f = fields[ UNBOX(fieldHandle) ];
  ILType *type = ILField_Type(f);
  return boxType(env, cls, type);
}

JNIEXPORT jstring JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_resolveStringToken
  (JNIEnv *env,
   jobject cls, 
   jobject methodHandle,
   jint tok) 
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILUInt32 token = *((ILUInt32 *) &tok);
  ILMethod *method = methods[ UNBOX(methodHandle) ];

  ILUInt32 stringLength;
  const char *str =
    ILImageGetUserString(ILProgramItem_Image(method),
			 token&~IL_META_TOKEN_MASK, 
			 &stringLength);
       
  return env->NewString((const jchar *)str, stringLength);
}

JNIEXPORT jobjectArray JNICALL
 Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetHandlers
  (JNIEnv *env, 
   jobject cls, 
   jobject methodObj,
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }

  TRY(exp, env)

  ILMethod *method = methods[ UNBOX(methodHandle) ];
  ILMethodCode code;

  if(!ILMethodGetCode(method, &code)) {
    THROW(exp, "no code for method");
  }

  jclass handlerCls = env->FindClass("com/ibm/wala/dotnet/loader/CLRMethod$ExceptionEntry");
  THROW_ANY_EXCEPTION(exp);

  ILException *exceptions;
  if(!ILMethodGetExceptions(method, &code, &exceptions)) {

    jobjectArray none = env->NewObjectArray(0, handlerCls, NULL);
    THROW_ANY_EXCEPTION(exp);

    return none;
  }

  int count = 0;
  ILException *ptr = exceptions;
  while (ptr != NULL) {
    ptr = ptr->next;
    count++;
  }

  jobjectArray data = env->NewObjectArray(count, handlerCls, NULL);
  THROW_ANY_EXCEPTION(exp);

  jmethodID ctor = env->GetMethodID(handlerCls, "<init>", "(Lcom/ibm/wala/dotnet/loader/CLRMethod;Ljava/lang/String;IIIIII)V");
  THROW_ANY_EXCEPTION(exp);

  int idx = 0;
  ptr = exceptions;
  while (ptr != NULL) {
    int catchTypeToken = -1;
    int filterOffset = -1;
    const char *handlerType;
    if ((ptr->flags & IL_META_EXCEPTION_FILTER) != 0) {
      handlerType = "FILTER";
      filterOffset = ptr->extraArg;
    } else if ((ptr->flags & IL_META_EXCEPTION_FINALLY) != 0) {
      handlerType = "FINALLY";
    } else if ((ptr->flags & IL_META_EXCEPTION_FAULT) != 0) {
      handlerType = "FAULT";
    } else {
      handlerType = "CATCH";
      catchTypeToken = ptr->extraArg;
    }

    jstring handlerTypeStr = env->NewStringUTF(handlerType);
    THROW_ANY_EXCEPTION(exp);

    jobject elt = env->NewObject(handlerCls, ctor, methodObj, handlerTypeStr, catchTypeToken, ptr->tryOffset, ptr->tryLength, filterOffset, ptr->handlerOffset, ptr->handlerLength);
    THROW_ANY_EXCEPTION(exp);

    env->SetObjectArrayElement(data, idx++, elt);
    THROW_ANY_EXCEPTION(exp);

    ptr = ptr->next;
  }

  return data;

  CATCH()

  return NULL;
}

JNIEXPORT jobject JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_attributeGetType
  (JNIEnv *env, 
   jobject cls, 
   jobject attributeHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILAttribute *attr = attributes[ UNBOX(attributeHandle) ];
  if (ILAttributeTypeIsString(attr)) {
    return boxType(env, cls, ILType_String);
  } else {
    ILClass *classInfo;
    ILTypeSpec *spec;
    ILMethod *method;
    ILProgramItem *typeItem = ILAttributeTypeAsItem(attr);

    if (DEBUG) {
      fprintf(stderr, "attr type %p\n", typeItem);
    }

    if((spec = ILProgramItemToTypeSpec(typeItem)) != 0) {
      ILType *rawType = ILTypeSpec_Type(spec);
      return boxType(env, cls, rawType);
    } else if((classInfo = ILProgramItemToClass(typeItem)) != 0) {
      return BOX(typeId(classInfo));
    } else if((method = ILProgramItemToMethod(typeItem)) != 0) {
      classInfo = ILMethod_Owner(method);
      return BOX(typeId(classInfo));
    } else {
      return BOX(-1);
    }
  }
}

jobjectArray attribute_array(JNIEnv *env, jobject cls, ILProgramItem *item) {
  if (ILProgramItem_HasAttrs(item)) {
    jclass javaLangIntegerCls = env->FindClass("java/lang/Integer"); 
    int len = ILProgramItemNumAttributes(item);
    jobjectArray arr = 
      env->NewObjectArray(len, javaLangIntegerCls, NULL);
    
    int i = 0;
    ILAttribute *iter = NULL;
    do {
      iter = ILProgramItemNextAttribute(item, iter);
      if (iter == NULL) {
	break;
      }
      
      if (attributes.contains(iter)) {
	int id = attributes[iter];
	env->SetObjectArrayElement(arr, i++, BOX(id));
      } else {
	attributes[nextHandle] = iter;
	env->SetObjectArrayElement(arr, i++, BOX(nextHandle));
	nextHandle++;
      }
    } while (iter != NULL);
    
    return arr;

  } else {
    return NULL;
  }
}

JNIEXPORT jobjectArray JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_classGetCustomAttributes
  (JNIEnv *env,
   jobject cls,
   jobject classHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILClass *clss = classes[UNBOX(classHandle)];
  ILProgramItem *item = ILToProgramItem(clss);
  return attribute_array(env, cls, item);

}

JNIEXPORT jobjectArray JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_fieldGetCustomAttributes
  (JNIEnv *env, 
   jobject cls, 
   jobject fieldHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILField *field = fields[UNBOX(fieldHandle)];
  ILProgramItem *item = ILToProgramItem(field);
  return attribute_array(env, cls, item);

}

JNIEXPORT jobjectArray JNICALL 
Java_com_ibm_wala_dotnet_loader_dotgnu_DotGnuInterface_methodGetCustomAttributes
  (JNIEnv *env, 
   jobject cls,
   jobject methodHandle)
{
  if (DEBUG) { fprintf(stdout, "at %s, %d\n", __FILE__, __LINE__); fflush(stdout); }
  ILMethod *method = methods[UNBOX(methodHandle)];
  ILProgramItem *item = ILToProgramItem(method);
  return attribute_array(env, cls, item);
}
