using System;
using System.Collections.Generic;
using System.Text;

namespace AnalysisScopeGenerator
{
    public class Factory
    {
        private class PrivateScope
        {
            private readonly System.Reflection.Assembly asm;
            private readonly System.Collections.Generic.IList<System.Reflection.Assembly> transitiveDependencies;

            public PrivateScope(System.Reflection.Assembly asm,
                System.Collections.Generic.IList<System.Reflection.Assembly> transitiveDependencies)
            {
                this.asm                    = asm;
                this.transitiveDependencies = transitiveDependencies;
            }

            public System.Reflection.Assembly Assembly
            {
                get
                {
                    return asm;
                }
            }

            public bool Subsumes(PrivateScope other)
            {
                return transitiveDependencies.Contains(other.asm);
            }

            public static IList<PrivateScope> GetMinCover(IList<PrivateScope> scopes)
            {
                IList<PrivateScope> result = new List<PrivateScope>();
                foreach (PrivateScope ps in scopes)
                {
                    Boolean isExcluded = false;
                    foreach (PrivateScope other in scopes)
                    {
                        if (!other.Equals(ps))
                        {
                            if (other.Subsumes(ps))
                            {
                                isExcluded = true;
                                break;
                            }
                        }
                    }
                    if (!isExcluded)
                    {
                        result.Add(ps);
                    }
                }
                return result;
            }

            public System.Collections.Generic.IList<System.Reflection.Assembly> Dependencies
            {
                get 
                { 
                    return transitiveDependencies; 
                }
            } 

            public override bool Equals(object obj)
            {
                if (obj is PrivateScope)
                {
                    PrivateScope other = obj as PrivateScope;
                    return ((this.asm.Equals(other.asm)) && (this.transitiveDependencies.Equals(other.transitiveDependencies)));
                }
                return false;
            }

            public override int GetHashCode()
            {
                return (asm.GetHashCode() + transitiveDependencies.GetHashCode());
            }
        }

        static Factory()
        {
            System.AppDomain.CurrentDomain.AssemblyResolve += new ResolveEventHandler(CurrentDomain_AssemblyResolve);
        }

        static System.Reflection.Assembly CurrentDomain_AssemblyResolve(object sender, ResolveEventArgs args)
        {
            Console.WriteLine(new StringBuilder("Could not resolve assembly: ").Append(args.Name).Append("."));
            return null;
        }

        public static bool MakeScope(String scopeFilePath, String sysAssemblyPath, params String[] appAssemblyPaths)
        {
            return MakeScope(scopeFilePath, null, sysAssemblyPath, appAssemblyPaths);
        }

        public static bool MakeScope(String scopeFilePath, String metascopeFilePath, String sysAssemblyPath, params String[] appAssemblyPaths)
        {
            return MakeScope(scopeFilePath, null, sysAssemblyPath, appAssemblyPaths, null);
        }

        public static bool MakeScope(String scopeFilePath, String metascopeFilePath, String sysAssemblyPath, String[] appAssemblyPaths, String[] refAssemblyPaths)
        {
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> libAssemblies = LoadAssemblies(refAssemblyPaths);
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> appAssemblies = LoadAssemblies(appAssemblyPaths);
            System.Reflection.Assembly sysAssembly = System.Reflection.Assembly.ReflectionOnlyLoadFrom(sysAssemblyPath);
            libAssemblies.Add(sysAssembly.FullName, sysAssembly);
            
            System.Collections.Generic.IList<String> visited = new System.Collections.Generic.List<String>(appAssemblyPaths.Length);
            System.Collections.Generic.IList<System.Reflection.Assembly> appAssembliesAsList =
                new System.Collections.Generic.List<System.Reflection.Assembly>(appAssemblies.Values);
            if (metascopeFilePath != null)
            {
                if (System.IO.File.Exists(metascopeFilePath))
                {
                    System.IO.File.Delete(metascopeFilePath);
                }
            }
            System.Collections.Generic.IList<PrivateScope> privateScopes = new System.Collections.Generic.List<PrivateScope>(appAssemblies.Count);
            foreach (System.Reflection.Assembly appAsm in appAssembliesAsList)
            {
                if (AddToScope(appAsm, appAssemblies, libAssemblies, visited))
                {
                    if (metascopeFilePath != null)
                    {
                        CreatePrivateScope(appAsm, sysAssembly, appAssemblies, libAssemblies, metascopeFilePath, privateScopes);
                    }
                }
            }
            /* 
             * We need to be careful not to include in the scope library assemblies which are not referenced by an application assembly.
             * This can happen, e.g., with the classpath-like assemblies.
             */
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> referencedLibAssemblies =
                new Dictionary<String, System.Reflection.Assembly>(libAssemblies.Count);
            foreach (String asmFullName in visited)
            {
                if (!appAssemblies.ContainsKey(asmFullName))
                {
                    if (!referencedLibAssemblies.ContainsKey(asmFullName))
                    {
                        referencedLibAssemblies.Add(asmFullName, libAssemblies[asmFullName]);
                    }
                }
            }
            WriteGlobalScope(scopeFilePath, sysAssembly, appAssemblies, referencedLibAssemblies);
            if (metascopeFilePath != null)
            {
                RegisterPrivateScopes(metascopeFilePath, PrivateScope.GetMinCover(privateScopes), sysAssembly, appAssemblies, libAssemblies);
            }
            return true;
        }

        private static System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> LoadAssemblies(String[] paths)
        {
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> result = new System.Collections.Generic.Dictionary<String, System.Reflection.Assembly>((paths == null) ? 1 : paths.Length);
            if (paths != null)
            {
                for (int index = 0; index < paths.Length; ++index)
                {
                    try
                    {
                        System.Reflection.Assembly asm = System.Reflection.Assembly.ReflectionOnlyLoadFrom(paths[index]);
                        if (!result.ContainsKey(asm.FullName))
                            result.Add(asm.FullName, asm);
                    }
                    catch (BadImageFormatException)
                    {
                        Console.WriteLine(new StringBuilder("ERROR: Could not load assembly at: ").Append(paths[index]));
                    }
                    catch (System.IO.FileLoadException)
                    {
                        Console.WriteLine(new StringBuilder("ERROR: Could not load assembly at: ").Append(paths[index]));
                    }
                }
            }
            return result;
        }

        private static void WriteGlobalScope(String scopeFilePath, System.Reflection.Assembly sysAssembly,
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> appAssemblies,
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> libAssemblies)
        {
            using (System.IO.TextWriter tw = new System.IO.StreamWriter(scopeFilePath))
            {
                tw.Write("lib,");
                tw.Write(sysAssembly.Location);
                tw.WriteLine();
                foreach (System.Reflection.Assembly appAsm in appAssemblies.Values)
                {
                    tw.Write("app,");
                    tw.Write(appAsm.Location);
                    tw.WriteLine();
                }
                foreach (System.Reflection.Assembly libAsm in libAssemblies.Values)
                {
                    if (!libAsm.Equals(sysAssembly))
                    {
                        tw.Write("lib,");
                        tw.Write(libAsm.Location);
                        tw.WriteLine();
                    }
                }
            }
        }

        private static System.Collections.Generic.IList<System.Reflection.Assembly> ComputeTransitiveDependencies(
            System.Reflection.Assembly asm, 
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> appAssemblies,
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> libAssemblies,
            System.Collections.Generic.IList<String> visited)
        {
            System.Collections.Generic.IList<System.Reflection.Assembly> result =
                new System.Collections.Generic.List<System.Reflection.Assembly>(1);
            result.Add(asm);
            visited.Add(asm.FullName);
            foreach (System.Reflection.AssemblyName asmName in asm.GetReferencedAssemblies())
            {
                String fullName = asmName.FullName;
                if (!visited.Contains(fullName))
                {
                    if (appAssemblies.ContainsKey(fullName))
                    {
                        foreach (System.Reflection.Assembly a in ComputeTransitiveDependencies(appAssemblies[fullName], appAssemblies, libAssemblies, visited))
                        {
                            result.Add(a);
                        }
                    }
                    else if (libAssemblies.ContainsKey(fullName))
                    {
                        foreach (System.Reflection.Assembly a in ComputeTransitiveDependencies(libAssemblies[fullName], appAssemblies, libAssemblies, visited))
                        {
                            result.Add(a);
                        }
                    }
                    else
                    {
                        throw new System.ArgumentException("ERROR: expected all dependencies to be resolved!");
                    }
                }
            }
            return result;
        }

        private static void RegisterPrivateScopes(String metascopePath, System.Collections.Generic.IList<PrivateScope> privateScopes,
            System.Reflection.Assembly sysAssembly,
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> appAssemblies,
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> libAssemblies)
        {
            foreach (PrivateScope ps in privateScopes)
            {
                System.Reflection.Assembly appAsm = ps.Assembly;
                System.Collections.Generic.IList<System.Reflection.Assembly> transitiveDependencies = ps.Dependencies;
                String path = System.IO.Path.GetTempFileName();
                using (System.IO.TextWriter tw = new System.IO.StreamWriter(path))
                {
                    tw.Write("lib,");
                    tw.Write(sysAssembly.Location);
                    tw.WriteLine();

                    tw.Write("app,");
                    tw.Write(appAsm.Location);
                    tw.WriteLine();

                    foreach (System.Reflection.Assembly asm in transitiveDependencies)
                    {
                        String fullName = asm.FullName;
                        if (appAssemblies.ContainsKey(fullName))
                        {
                            if (!asm.Equals(appAsm))
                            {
                                tw.Write("app,");
                                tw.Write(asm.Location);
                                tw.WriteLine();
                            }
                        }
                        else if (libAssemblies.ContainsKey(fullName))
                        {
                            if (!asm.Equals(sysAssembly))
                            {
                                tw.Write("lib,");
                                tw.Write(asm.Location);
                                tw.WriteLine();
                            }
                        }
                        else
                        {
                            throw new System.ArgumentException("ERROR: expected all dependencies to be resolved!");
                        }
                    }
                }
                using (System.IO.TextWriter tw = new System.IO.StreamWriter(metascopePath, true))
                {
                    tw.WriteLine(path);
                }
            }
        }

        private static void CreatePrivateScope(System.Reflection.Assembly appAsm, 
            System.Reflection.Assembly sysAssembly,
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> appAssemblies,
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> libAssemblies,
            String metascopePath,
            System.Collections.Generic.IList<PrivateScope> privateScopes)
        {
            System.Collections.Generic.IList<String> visited = new System.Collections.Generic.List<String>(1);
            System.Collections.Generic.IList<System.Reflection.Assembly> transitiveDependencies =
                ComputeTransitiveDependencies(appAsm, appAssemblies, libAssemblies, visited);
            privateScopes.Add(new PrivateScope(appAsm, transitiveDependencies));
        }

        private static Boolean AddToScope(System.Reflection.Assembly asm,
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> appAssemblies,
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> libAssemblies,
            System.Collections.Generic.IList<String> visited)
        {
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> sandboxedLibs = new System.Collections.Generic.Dictionary<String, System.Reflection.Assembly>();
            System.Collections.Generic.IList<String> sandboxedVisited = new System.Collections.Generic.List<String>(appAssemblies.Count);
            Boolean result = AddToSandbox(asm, appAssemblies, sandboxedLibs, sandboxedVisited);
            if (result)
            {
                foreach (String libFullName in sandboxedLibs.Keys)
                {
                    if (!libAssemblies.ContainsKey(libFullName))
                    {
                        libAssemblies.Add(libFullName, sandboxedLibs[libFullName]);
                    }
                }
                foreach (String v in sandboxedVisited)
                {
                    if (!visited.Contains(v))
                    {
                        visited.Add(v);
                    }
                }
            }
            else
            {
                appAssemblies.Remove(asm.FullName);
                Console.WriteLine(new StringBuilder("ERROR: Could not load assembly ").Append(asm.FullName).Append(" due to bad or missing dependencies."));
            }
            return result;
        }

        private static Boolean AddToSandbox(System.Reflection.Assembly asm,
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> appAssemblies,
            System.Collections.Generic.IDictionary<String, System.Reflection.Assembly> libAssemblies,
            System.Collections.Generic.IList<String> visited)
        {
            Boolean result = true;
            try
            {
                visited.Add(asm.FullName);
                foreach (System.Reflection.AssemblyName asmName in asm.GetReferencedAssemblies())
                {
                    if (!visited.Contains(asmName.FullName))
                    {
                        System.Reflection.Assembly refAsm;
                        if (appAssemblies.ContainsKey(asmName.FullName))
                        {
                            refAsm = appAssemblies[asmName.FullName];
                        }
                        else if (libAssemblies.ContainsKey(asmName.FullName))
                        {
                            refAsm = libAssemblies[asmName.FullName];
                        }
                        else
                        {
                            refAsm = System.Reflection.Assembly.ReflectionOnlyLoad(asmName.FullName);
                            libAssemblies.Add(asmName.FullName, refAsm);
                        }
                        result &= AddToSandbox(refAsm, appAssemblies, libAssemblies, visited);
                    }
                }
            }
            catch (System.IO.FileLoadException)
            {
                result = false;
            }
            catch (System.IO.FileNotFoundException)
            {
                result = false;
            }
            catch (System.ArgumentNullException)
            {
                result = false;
            }
            catch (System.BadImageFormatException)
            {
                result = false;
            }
            return result;
        }
    }
}
