using System;
using System.Collections.Generic;
using System.Text;
using System.Reflection;

namespace CSharpImpl
{
    public class ClassInfo
    {
        private readonly Type type;
        private Vector<ClassInfo> interfaces;
        private Vector<FieldInfo> instanceFields;
        private Vector<FieldInfo> staticFields;
        private Vector<MethodInfo> instanceMethods;
        private Vector<MethodInfo> staticMethods;
        private MethodInfo staticConstructor;

        #region CTOR(s)
        public ClassInfo(Type classType)
        {
            //System.Diagnostics.Debug.Assert(classType.IsClass);
            this.type = classType; 
        }
        #endregion

        #region Methods
		public PropertyInfo getPropertyForMethod(MethodBase method) {
			try {
				if (type.GetProperties() != null) {
					foreach (PropertyInfo p in type.GetProperties()) {
						if (method.Equals(p.GetGetMethod()) || method.Equals(p.GetSetMethod())) {
							return p;
						}
					}
				}
			} catch (TypeLoadException e) {
	
			}
			return null;
		}
		
		public Vector<FieldInfo> ComputeFields(System.Reflection.BindingFlags bindingFlags)
        { 
            System.Reflection.FieldInfo[] fields    = type.GetFields(bindingFlags);
            Vector<FieldInfo> result                = new Vector<FieldInfo>(fields.Length);
            foreach (System.Reflection.FieldInfo field in fields)
            {
                result.Add(new FieldInfo(field));
            }
            return result;
        }

        public Vector<MethodInfo> ComputeConstructors()
        {
            System.Reflection.ConstructorInfo[] publicCtors     = type.GetConstructors(System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.Public);
            System.Reflection.ConstructorInfo[] nonPublicCtors  = type.GetConstructors(System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
            Vector<MethodInfo> result = new Vector<MethodInfo>(publicCtors.Length + nonPublicCtors.Length);
            foreach (System.Reflection.ConstructorInfo ctor in publicCtors)
            {
				try {
                	result.Add(new MethodInfo(ctor, type));
				} catch (MissingMethodException e) {
					// why does this happen?
				}
            }
            foreach (System.Reflection.ConstructorInfo ctor in nonPublicCtors)
            {
				try {
                	result.Add(new MethodInfo(ctor, type));
				} catch (MissingMethodException e) {
					// why does this happen?
				}
            }
            return result;
        }
        
        

        public Vector<MethodInfo> ComputeMethods(System.Reflection.BindingFlags bindingFlags)
        {
            System.Reflection.MethodInfo[] methods  = type.GetMethods(bindingFlags);
            Vector<MethodInfo> result               = new Vector<MethodInfo>(methods.Length);
            foreach (System.Reflection.MethodInfo method in methods)
            {
				try {
                	result.Add(new MethodInfo(method, type));
				} catch (MissingMethodException e) {
					// why does this happen?
				}
            }
            return result;
        }

        public override int GetHashCode()
        {
            if (type == null)
                return 0;
            else
                return type.GetHashCode();
        }

        public override Boolean Equals(Object other)
        {
            if (other is ClassInfo)
            {
                ClassInfo otherCI = (ClassInfo) other;
                if (this.type == null)
                    return (otherCI.type == null);
                else
                    return this.type.Equals(otherCI.type);
            }
            return false;
        }
        #endregion

        #region Properties
        public Vector<AttributeInfo> Attributes
        {
            get
            { 
                IList<System.Reflection.CustomAttributeData> attributes = System.Reflection.CustomAttributeData.GetCustomAttributes(type);
                Vector<AttributeInfo> result                            = new Vector<AttributeInfo>(attributes.Count);
                foreach (System.Reflection.CustomAttributeData att in attributes)
                {
                    result.Add(new AttributeInfo(att));
                }
                return result;
            }
        }

        public Vector<ClassInfo> Interfaces
        {
            get
            {
                if (interfaces == null)
                {
                    Type[] interfaceTypes   = type.GetInterfaces();
                    interfaces              = new Vector<ClassInfo>(interfaceTypes.Length);
                    foreach (Type interfce in interfaceTypes)
                    {
                        interfaces.Add(new ClassInfo(interfce));
                    }
                }
                return interfaces;
            }
        }

        public Vector<MethodInfo> InstanceMethods
        {
            get
            {
                if (instanceMethods == null)
                {
                    instanceMethods = new Vector<MethodInfo>(1);
                    instanceMethods.AddAll(ComputeMethods(System.Reflection.BindingFlags.DeclaredOnly | System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.Public));
                    instanceMethods.AddAll(ComputeMethods(System.Reflection.BindingFlags.DeclaredOnly | System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic));
                    instanceMethods.AddAll(ComputeConstructors());
                }
                return instanceMethods;
            }
        }

        public MethodInfo StaticConstructor
        {
            get
            {
                if (staticConstructor == null  && type.TypeInitializer != null)
                {
                    staticConstructor =  new MethodInfo(type.TypeInitializer, type);
                }
                return staticConstructor;
            }
        }
        
        public Vector<MethodInfo> StaticMethods
        {
            get
            {
                if (staticMethods == null)
                {
                    staticMethods = new Vector<MethodInfo>(1);
                    staticMethods.AddAll(ComputeMethods(System.Reflection.BindingFlags.DeclaredOnly | System.Reflection.BindingFlags.Static | System.Reflection.BindingFlags.Public));
                    staticMethods.AddAll(ComputeMethods(System.Reflection.BindingFlags.DeclaredOnly | System.Reflection.BindingFlags.Static | System.Reflection.BindingFlags.NonPublic));
                }
                return staticMethods;
            }
        }

        public Vector<FieldInfo> InstanceFields
        {
            get
            {
                if (instanceFields == null)
                {
                    instanceFields = new Vector<FieldInfo>(1);
                    instanceFields.AddAll(ComputeFields(System.Reflection.BindingFlags.DeclaredOnly | System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.Public));
                    instanceFields.AddAll(ComputeFields(System.Reflection.BindingFlags.DeclaredOnly | System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic));
                }
                return instanceFields;
            }
        }

        public Vector<FieldInfo> StaticFields
        {
            get
            {
                if (staticFields == null)
                {
                    staticFields = new Vector<FieldInfo>(1);
                    staticFields.AddAll(ComputeFields(System.Reflection.BindingFlags.DeclaredOnly | System.Reflection.BindingFlags.Static | System.Reflection.BindingFlags.Public));
                    staticFields.AddAll(ComputeFields(System.Reflection.BindingFlags.DeclaredOnly | System.Reflection.BindingFlags.Static | System.Reflection.BindingFlags.NonPublic));
                }
                return staticFields;
            }
        }

        public Type Type
        {
            get
            { 
                return this.type;
            }
        }

        public String Name
        {
            get
            {   
				string name = (type.FullName == null) ? (type.Namespace + "/" + type.Name) : type.FullName;
           		if (name.IndexOf('`') > 1) {
					return name.Substring(0, name.IndexOf('`')).Replace('.', '/') + name.Substring(name.IndexOf('`'));
				} else {
					return name.Replace('.', '/');
				}
			}
        }

        public Boolean IsValueType
        {
            get
            {
                return type.IsValueType;
            }
        }

        public Boolean IsAbstract
        {
            get
            {
                return type.IsAbstract;
            }
        }

        public Boolean IsByRef
        {
            get
            {
                return type.IsByRef;
            }
        }

        public Boolean IsGeneric
        {
            get
            {
                return (type.IsGenericType && !type.IsGenericTypeDefinition && !type.IsGenericParameter);
            }
        }

        public Boolean IsGenericParameter
        {
            get
            {
                return type.IsGenericParameter;
            }
        }

        public Boolean IsPrimitive
        {
            get
            {
                return type.IsPrimitive || type.Equals(typeof(void));
            }
        }

        public Boolean IsPublic
        {
            get
            {
                return type.IsPublic;
            }
        }

        public Boolean IsArray
        {
            get
            {
                return type.IsArray;
            }
        }

        public Boolean IsInterface
        {
            get
            {
                return type.IsInterface;
            }
        }

        public Boolean IsPointer
        {
            get
            {
                return type.IsPointer;
            }
        }

        public ClassInfo Parent
        {
            get
            {
                Type baseType = type.BaseType;
                return (baseType == null) ? null : new ClassInfo(baseType);
            }
        }
        #endregion
    }
}
