#  (c) Copyright IBM Corp. 1991, 2016. All Rights Reserved.
#  (c) Copyright HCL Technologies Ltd. 2016, 2024 All Rights Reserved.
#
#  Dast Proxy Server Demo Script
#  --------------------------------
#  This script is a sample workflow of using the Dast Proxy Server.
#  For running this script Python must be installed,
#  and the Python library "Requests: HTTP for Humans" must also
#  be installed (it can be found at: http://docs.python-requests.org).
#
# This demo script starts the proxy, sends a request through the proxy,
# stops the proxy, and downloads the recorded traffic file.
#

import re
import os
import subprocess
import json
import requests
from time import sleep


class ConfigData:
    def __init__(self):
        ########   Please fill out the following details:   ########
        self.proxy_server_port = "8383"
        self.proxy_server_domain = "localhost"
        self.proxy_port = "20000"
        ############################################################

        ########   Please fill out the following details for ASoC upload example:   ########
        self.web_application_username = "jsmith"
        self.web_application_password = "demo1234"
        self.asoc_base_url = "https://cloud.appscan.com:443"
        self.asoc_key_id = ""
        self.asoc_key_secret = ""
        self.asoc_application_id = "1"
        #If pss isn't required then leave the presence id blank ("")
        self.asoc_presence_id = ""
        self.asoc_scan_starting_point_url = "http://demo.testfire.net/"
        self.asoc_scan_name = "my demo scan"
        ############################################################


class ProxyServer:
    def __init__(self, config: ConfigData):
        self.config = config
        self.process = None;
        self.base_url = "http://" + self.config.proxy_server_domain + ":" + config.proxy_server_port + "/automation/"

    def is_proxy_server_running(self):
        try:
            response = requests.get(self.base_url, verify=False)
        except:
            return False
        return response != None and response.status_code == 200

    def download_root_ca(self):
        print("** Downloading self signed root certificate (needed if we are working with https and want to avoid SSL errors)")
        url = self.base_url + "Certificate/"
        response = requests.get(url, verify=False)
        pem_root_ca_str = response.content.decode("utf-8")
        pem_root_ca_file_name = "rootCaPuKey.pem"
        pem_file = open(pem_root_ca_file_name, "w")
        pem_file.write(pem_root_ca_str)
        pem_file.close()
        print("*** Self signed root certificate has been saved as file '" + pem_root_ca_file_name + "'")

    def start_proxy(self):
        print("** Starting proxy on port '%s'" % (self.config.proxy_port))
        url = self.base_url + "StartProxy/" + self.config.proxy_port
        response = requests.get(url, verify=False)
        print("*** Proxy Server Response:" + str(response.json()))

    def stop_proxy(self):
        print("** Stopping proxy")
        url = self.base_url + "StopProxy/" + self.config.proxy_port
        response = requests.get(url, verify=False)
        print("*** Proxy Server Response:" + str(response.json()))

    def download_traffic(self):
        print("** Downloading the traffic file")
        url = self.base_url + "Traffic/" + self.config.proxy_port
        response = requests.get(url, verify=False)
        self.config.traffic_file_name = "scan.dast.config"
        traffic_file = open(self.config.traffic_file_name, "wb")
        for chunk in response.iter_content(chunk_size=1024):
            if chunk:
                traffic_file.write(chunk)
        traffic_file.close()
        print("*** The traffic has been saved as file '" + self.config.traffic_file_name + "'")


class PublishSettingsData:
    def __init__(self, config: ConfigData):
        self.KeyId = config.asoc_key_id
        self.KeySecret = config.asoc_key_secret
        self.ScanName = config.asoc_scan_name
        self.Stp = config.asoc_scan_starting_point_url
        self.AppId = config.asoc_application_id
        self.AppUserName = config.web_application_username
        self.AppPassword = config.web_application_password

class AsocLoginWithKeyApiData:
    def __init__(self, config: ConfigData):
        self.KeyId = config.asoc_key_id
        self.KeySecret = config.asoc_key_secret

class CreateScanWithFileData:
    def __init__(self, config: ConfigData):
        self.TestOnly = True
        # If the PSS (asoc_presence_id) is NOT an empty string, then add the element
        if bool(config.asoc_presence_id):
            self.PresenceId = config.asoc_presence_id
        self.ScanName = config.asoc_scan_name
        self.EnableMailNotification = False
        self.AppId = config.asoc_application_id
        self.ScanConfiguration = { 
            'Target': { 
                'StartingUrl': config.asoc_scan_starting_point_url  
            }, 
            'Login': { 
                'UserName': config.web_application_username ,
                'Password': config.web_application_password
            },
        }
        self.ExploreItems = [
            {
                'FileId': config.traffic_file_id ,
                'MultiStep': False
            }
        ]


class AsocRestApi:
    def __init__(self, config: ConfigData):
        self.config = config

    def getHeaders(self, withJsonContentType, withToken):
        if withJsonContentType and withToken:
            return {
                'Content-type': 'application/json',
                'Authorization': 'Bearer ' + self.asoc_token
            }
        elif withJsonContentType:
            return { 'Content-type': 'application/json' }
        else:
            return { 'Authorization': 'Bearer ' + self.asoc_token }

    def loginWithKeyId(self):
        print("** Login into ASoC with API Key")
        url = self.config.asoc_base_url + '/api/v4/Account/ApiKeyLogin'
        requestData = AsocLoginWithKeyApiData(self.config)
        requestBody = json.dumps(requestData.__dict__)
        response = requests.post(url=url, data=requestBody, headers=self.getHeaders(True, False), verify=False)
        response_str = str(response.json())
        print("ASoC Server Response:" + response_str)
        if (response.status_code == 200 and response_str.__len__() > 0):
            self.asoc_token = response.json()['Token']
            print("*** Login has finished successfully")
        else:
            print("XX Failed to login into ASoC")
            exit(1)

    def uploadTrafficFile(self):
        print("** Uploading traffic dast.config file to ASoC")
        url = self.config.asoc_base_url + '/api/v4/FileUpload'
        files = {'Authorization' : (None, self.asoc_token, 'text/plain'), 'uploadedFile': (self.config.traffic_file_name, open(self.config.traffic_file_name, 'rb').read())}
        response = requests.post(url=url, headers=self.getHeaders(False, True), files=files, verify=False)
        response_str = str(response.json())
        print("ASoC Server Response:" + response_str)
        if (response.status_code == 200 and response_str.__len__() > 0):
            self.config.traffic_file_id = response.json()['FileId']
            print("*** Traffic file has been uploaded successfully")
        else:
            print("XX Failed to upload file to ASoC")
            exit(1)

    def createNewScanWithTraffic(self):
        print("** Publishing scan with the recorded traffic to ASoC")
        url = self.config.asoc_base_url + '/api/v4/Scans/Dast'
        requestData = CreateScanWithFileData(self.config)
        requestBody = json.dumps(requestData.__dict__)
        response = requests.post(url=url, data=requestBody, headers=self.getHeaders(True, True), verify=False)
        response_str = str(response.json())
        print("ASoC Server Response:" + response_str)
        if (response.status_code == 201 and response_str.__len__() > 0):
            print("*** Scan with the recorded traffic has been published into ASoC successfully")
        else:
            print("XX Failed to create scan in ASoC")
            exit(1)


def main():
    config = ConfigData()
    proxy_server = ProxyServer(config)
    asoc_rest_api = AsocRestApi(config)

    is_running = proxy_server.is_proxy_server_running()
    if is_running:
        proxy_server.download_root_ca()
        proxy_server.start_proxy()
        run_traffic_script(config.proxy_port)
        proxy_server.stop_proxy()
        proxy_server.download_traffic()
        #Now that we have the traffic file, and we can use it with ASoC REST API or with ASE REST API
        asoc_rest_api.loginWithKeyId()
        asoc_rest_api.uploadTrafficFile()
        asoc_rest_api.createNewScanWithTraffic()
    else:
        print("XX Proxy Server wasn't found on port '" + config.proxy_server_port + "'")


def run_traffic_script(proxy_port):
    #This function should start the script\program of sending the traffic through the proxy port (it can be a selenium script or in any other way)
    #input("Start manual browsing and press any key to stop the traffic recording...\n")
    print("** Sending a GET request through the proxy to the url 'http://demo.testfire.net/'")
    http_proxy = "http://localhost:" + proxy_port
    https_proxy = "https://localhost:" + proxy_port
    ftp_proxy = "ftp://localhost:" + proxy_port
    proxyDict = {
        "http": http_proxy,
        "https": https_proxy,
        "ftp": ftp_proxy
    }
    response = requests.get("http://demo.testfire.net/", proxies=proxyDict)
    print("*** Finished running traffic through the proxy")


main()